<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\File;
use App\Models\Project;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class FileController extends Controller
{

    // Subir un archivo
    public function upload(Request $request)
    {
        $request->validate([
            'file' => 'required|file|max:10240', // Máximo 10MB
            'storage' => 'required|string',
        ]);

        // Generar UUID
        $uuid = Str::uuid()->toString();

        // Obtener el proyecto autorizado
        $project = $this->getAuthorizedProject($request->header('Authorization'));

        // Directorio del proyecto designado en `folder_path`
        $directory = $project->folder_path . '/' . $request->storage;

        // Obtener el archivo
        $file = $request->file('file');

        // Construir el nuevo nombre del archivo
        $newFileName = $uuid . '.' . $file->getClientOriginalExtension();

        // Guardar el archivo con el nuevo nombre
        $path = $file->storeAs($directory, $newFileName);

        // Guardar los datos del archivo en la base de datos
        $fileRecord = File::create([
            'name' => $file->getClientOriginalName(), // Nombre original
            'path' => $path, // Ruta relativa en el almacenamiento
            'uuid' => $uuid, // UUID generado
            'size' => $file->getSize(), // Tamaño del archivo
            'type' => $file->getClientOriginalExtension(), // Extensión del archivo
            'url' => url("storage/{$path}"), // URL completa del archivo
            'project_id' => $project->id, // ID del proyecto
        ]);

        return response()->json([
            'message' => 'Archivo subido correctamente.',
            'file' => $fileRecord,
        ], 201);
    }

    // Mostrar un archivo
    public function show($uuid)
    {
        $file = File::where('uuid', $uuid)->first();

        if (!$file) {
            return response()->json(['message' => 'Archivo no encontrado.'], 404);
        }

        $project = $this->getAuthorizedProject(request()->header('Authorization'));

        if ($file->project_id !== $project->id) {
            return response()->json(['message' => 'No Autorizado para ver este Archivo.'], 403);
        }

        if (!Storage::exists($file->path)) {
            return response()->json(['message' => 'Archivo no encontrado en el servidor.'], 404);
        }

        return response()->json([
            'file' => $file,
            'message' => 'Archivo obtenido correctamente.',
        ]);
    }

    // Eliminar un archivo
    public function destroy($uuid)
    {
        try {
            $file = File::where('uuid', $uuid)->first();

            if (!$file) {
                return response()->json(['message' => 'Archivo no encontrado.'], 404);
            }

            // Obtener el proyecto autorizado
            $project = $this->getAuthorizedProject(request()->header('Authorization'));

            // Verificar si el archivo pertenece al proyecto autorizado
            if ($file->project_id != $project->id) {
                return response()->json(['message' => 'No Autorizado para eliminar este Archivo.'], 403);
            }

            // Eliminar archivo físico si existe
            if (Storage::exists($file->path)) {
                Storage::delete($file->path);
            }

            // Eliminar registro del archivo de la base de datos
            $file->delete();

            return response()->json([
                'message' => 'Archivo eliminado correctamente.',
                'file' => $file
            ], 200);
        } catch (\Exception $e) {
            // Manejo de excepciones
            return response()->json(['message' => 'Error al eliminar el archivo: ' . $e->getMessage()], 500);
        }
    }

    // Función auxiliar para obtener el proyecto autorizado
    private function getAuthorizedProject($token)
    {
        $project = Project::where('token', $token)->where('status', 'active')->first();

        if (!$project) {
            abort(response()->json(['message' => 'Token Invalido o Proyecto Inactivo.'], 403));
        }

        return $project;
    }
}
