<?php

namespace App\Traits;

use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

trait HasUuidAndSlug
{
    protected static function bootHasUuidAndSlug(): void
    {
        // Auto-generar UUID al crear
        static::creating(function ($model) {
            if (Schema::hasColumn($model->getTable(), 'uuid') && empty($model->uuid)) {
                $model->uuid = (string) Str::uuid();
            }

            // Auto-generar slug si el modelo lo define
            if ($model->shouldGenerateSlug()) {
                $model->generateSlug();
            }
        });

        // Actualizar slug si cambian los campos origen
        static::updating(function ($model) {
            if ($model->shouldGenerateSlug() && $model->hasSlugSourceChanged()) {
                $model->generateSlug();
            }
        });
    }

    protected function shouldGenerateSlug(): bool
    {
        return method_exists($this, 'sluggableString')
            && Schema::hasColumn($this->getTable(), 'slug');
    }

    protected function hasSlugSourceChanged(): bool
    {
        $sluggableFields = $this->getSlugSourceFields();

        foreach ($sluggableFields as $field) {
            if ($this->isDirty($field)) {
                return true;
            }
        }

        return false;
    }

    // Por defecto asume 'name', cada modelo sobrescribe si necesita
    protected function getSlugSourceFields(): array
    {
        return ['name'];
    }

    protected function generateSlug(): void
    {
        $slugString = $this->sluggableString();

        if (!empty($slugString)) {
            $this->slug = Str::slug($slugString);
        }
    }
}
