<?php

namespace App\Traits;

use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

trait HasApiTokenAuth
{
    public $plain_token; // Token sin hashear (solo en memoria, nunca se guarda en BD)

    // Generar token de acceso
    public function generateAccessToken(int $hoursValid = 2): void
    {
        $this->plain_token = Str::random(64);
        $this->access_token = hash('sha256', $this->plain_token);
        $this->token_expires_at = now()->addHours($hoursValid);
        $this->save();
    }

    // Renovar token existente (genera nuevo token y extiende expiración)
    public function renewToken(int $hoursValid = 2): void
    {
        $this->generateAccessToken($hoursValid);
    }

    // Invalidar token actual (logout)
    public function invalidateToken(): void
    {
        $this->access_token = null;
        $this->token_expires_at = null;
        $this->plain_token = null;
        $this->save();
    }

    // Verificar si el token está vigente
    public function isTokenValid(): bool
    {
        return $this->access_token
            && $this->token_expires_at
            && $this->token_expires_at->isFuture();
    }

    // Verificar contraseña
    public function checkPassword(string $password): bool
    {
        return Hash::check($password, $this->password);
    }

    // Establecer nueva contraseña
    public function setPassword(string $password): void
    {
        $this->password = Hash::make($password);
        $this->save();
    }

    // Mutador de contraseña
    public function setPasswordAttribute($value)
    {
        $this->attributes['password'] = Hash::make($value);
    }

    // Generar hash de contraseña sin guardar (útil para validaciones)
    public function hashPassword(string $password): string
    {
        return Hash::make($password);
    }

    // Actualizar timestamp de último acceso (NO toca token_expires_at)
    public function updateLastLogin(): void
    {
        $this->last_login_at = now();
        $this->save();
    }

    // Verificar si el token expirará pronto
    public function tokenWillExpireSoon(int $minutesThreshold = 30): bool
    {
        if (!$this->token_expires_at) {
            return true;
        }

        return $this->token_expires_at->diffInMinutes(now()) <= $minutesThreshold;
    }

    // Obtener tiempo restante del token en minutos
    public function getRemainingTokenMinutes(): ?int
    {
        if (!$this->token_expires_at) {
            return null;
        }

        $remaining = $this->token_expires_at->diffInMinutes(now(), false);
        return $remaining > 0 ? $remaining : 0;
    }

    // Verificar si tiene sesión activa
    public function hasActiveSession(): bool
    {
        return $this->isTokenValid();
    }

    // Extender expiración del token actual (sin generar nuevo token)
    public function extendTokenExpiration(int $hoursToAdd = 2): void
    {
        if ($this->access_token) {
            $this->token_expires_at = now()->addHours($hoursToAdd);
            $this->save();
        }
    }

    // Verificar si el usuario inició sesión recientemente
    public function hasLoggedInRecently(int $hours = 24): bool
    {
        return $this->last_login_at && $this->last_login_at->gte(now()->subHours($hours));
    }

    // Obtener el token plano desde el request actual
    public function getPlainTokenFromRequest(): ?string
    {
        return request()->bearerToken();
    }

    // Obtener el token plano sin hashear
    public function getPlainToken(): ?string
    {
        return $this->plain_token;
    }
}
