<?php

namespace App\Models;

use App\Enums\DocumentType;
use App\Enums\Gender;
use App\Enums\MaritalStatus;
use App\Traits\HasApiTokenAuth;
use App\Traits\HasRoles;
use App\Traits\HasUuidAndSlug;
use App\Traits\QueryScopeTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Hash;

class Student extends Model
{
    use HasFactory, HasApiTokenAuth, HasRoles, QueryScopeTrait, HasUuidAndSlug;

    protected $fillable = [
        'document_type',
        'document_number',
        'names',
        'paternal_surname',
        'maternal_surname',
        'birth_date',
        'birth_place_id',
        'marital_status',
        'gender',
        'languages',
        'phone_number',
        'whatsapp_number',
        'email',
        'residence_address',
        'residence_place_id',
        'school_graduation_year',
        'school_graduation_certificate',
        'school_id',
        'family_data',
        'document_file',
        'photo_student',
        'active',
        'validated_information',
        'password',
        'access_token',
        'token_expires_at',
        'last_login_at',
        'photo_profile',
    ];


    protected $hidden = [
        'password',
        'access_token',
    ];

    protected $appends = ['user_type', 'photo_profile_url'];

    protected $casts = [
        'document_type' => DocumentType::class,
        'gender' => Gender::class,
        'marital_status' => MaritalStatus::class,
        'birth_date' => 'date',
        'languages' => 'array',
        'family_data' => 'array',
        'token_expires_at' => 'datetime',
        'last_login_at' => 'datetime',
        'active' => 'boolean',
        'validated_information' => 'boolean',
    ];

    // Columnas donde se puede buscar (texto general)
    public array $searchable = ['document_number', 'names', 'paternal_surname', 'maternal_surname', 'email'];

    // Columnas por las que se puede ordenar
    public array $sortable = ['id', 'document_number', 'names', 'paternal_surname', 'maternal_surname', 'email'];

    // Relaciones que se pueden incluir
    public array $allowedIncludes = ['latest_enrollment_active', 'school', 'residenceDistrict', 'birthDistrict', 'enrollments'];

    // Columnas por las que se puede filtrar específicamente
    public array $filterable = ['document_number', 'names', 'paternal_surname', 'maternal_surname', 'email'];

    // Accessor para user_type (necesario para $appends)
    public function getUserTypeAttribute(): string
    {
        return 'student';
    }

    // HasUuidAndSlug trait: Slug
    protected function sluggableString(): string
    {
        return "{$this->document_number}-{$this->names}-{$this->paternal_surname}-{$this->maternal_surname}";
    }
    // HasUuidAndSlug trait: Slug
    protected function getSlugSourceFields(): array
    {
        return ['document_number', 'names', 'paternal_surname', 'maternal_surname'];
    }


    // Obtiene la URL completa de la foto de perfil
    public function getPhotoProfileUrlAttribute(): ?string
    {
        if (!$this->photo_profile) {
            return null;
        }

        // Si ya es una URL completa, retornarla
        if (filter_var($this->photo_profile, FILTER_VALIDATE_URL)) {
            return $this->photo_profile;
        }

        // Generar URL completa con el dominio del backend
        return url('storage/' . $this->photo_profile);
    }

    // Última matrícula activa
    public function latestEnrollmentActive()
    {
        return $this->hasOne(Enrollment::class)
            // ->where('active', true)
            ->whereIn('status', ['pending', 'finished', 'active'])
            ->latestOfMany();
    }

    // Relaciones
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function residenceDistrict()
    {
        return $this->belongsTo(District::class, 'residence_place_id');
    }

    public function birthDistrict()
    {
        return $this->belongsTo(District::class, 'birth_place_id');
    }

    public function enrollments()
    {
        return $this->hasMany(Enrollment::class);
    }
}
