<?php

namespace App\Models;

use App\Traits\QueryScopeTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\MorphToMany;

class Role extends Model
{
    use HasFactory, QueryScopeTrait;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'active',
    ];

    protected $casts = [
        'active' => 'boolean',
    ];

    // Columnas donde se puede buscar (texto general)
    public array $searchable = ['name', 'slug', 'description'];

    // Columnas por las que se puede ordenar
    public array $sortable = ['id', 'name', 'slug', 'created_at'];

    // Relaciones que se pueden incluir
    public array $allowedIncludes = ['permissions', 'users', 'teachers'];

    // Columnas por las que se puede filtrar específicamente
    public array $filterable = ['active', 'slug'];

    // Relación con permisos
    public function permissions(): BelongsToMany
    {
        return $this->belongsToMany(Permission::class, 'role_permissions')
            ->withTimestamps();
    }

    // Relación polimórfica inversa con usuarios
    public function users(): MorphToMany
    {
        return $this->morphedByMany(User::class, 'model', 'model_roles')
            ->withTimestamps();
    }

    public function teachers(): MorphToMany
    {
        return $this->morphedByMany(Teacher::class, 'model', 'model_roles')
            ->withTimestamps();
    }

    public function students(): MorphToMany
    {
        return $this->morphedByMany(Student::class, 'model', 'model_roles')
            ->withTimestamps();
    }

    // Asignar permisos al rol
    public function givePermissionTo(string|int|Permission|array $permissions): void
    {
        $permissions = is_array($permissions) ? $permissions : [$permissions];

        foreach ($permissions as $permission) {
            $permissionModel = $permission instanceof Permission
                ? $permission
                : Permission::where('slug', $permission)->orWhere('id', $permission)->firstOrFail();

            $this->permissions()->syncWithoutDetaching($permissionModel);
        }
    }

    // Remover permisos del rol
    public function revokePermissionTo(string|int|Permission|array $permissions): void
    {
        $permissions = is_array($permissions) ? $permissions : [$permissions];

        foreach ($permissions as $permission) {
            $permissionModel = $permission instanceof Permission
                ? $permission
                : Permission::where('slug', $permission)->orWhere('id', $permission)->first();

            if ($permissionModel) {
                $this->permissions()->detach($permissionModel);
            }
        }
    }

    // Verificar si el rol tiene un permiso
    public function hasPermission(string|Permission $permission): bool
    {
        $slug = $permission instanceof Permission ? $permission->slug : $permission;
        return $this->permissions()->where('slug', $slug)->exists();
    }
}
