<?php

namespace App\Models;

use App\Enums\EnrollmentStatus;
use App\Enums\GradeSchool;
use App\Enums\PaymentType;
use App\Traits\HasUuidAndSlug;
use App\Traits\QueryScopeTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Enrollment extends Model
{
    use HasFactory, QueryScopeTrait, HasUuidAndSlug;

    protected $fillable = [
        'uuid',
        'student_id',
        'cycle_id',
        'career_id',
        'status',
        'oath_declaration',
        'grade_school',
        'file_grade_school',
        'score',
        'classroom_id',
        'registration_steps',
        'information_sources',
        'total_amount',
        'payment_type',
        'discount_type_id',
        'discount_amount',
        'discount_file',
    ];

    protected $casts = [
        'status' => EnrollmentStatus::class,
        'grade_school' => GradeSchool::class,
        'payment_type' => PaymentType::class,
        'total_amount' => 'decimal:2',
        'score' => 'array',
        'information_sources' => 'array',
        'registration_steps' => 'array',
        'oath_declaration' => 'boolean',
        'discount_amount' => 'decimal:2',
    ];

    // Pasos requeridos para completar la matrícula
    public const REQUIRED_STEPS = [
        'personal_data',
        'additional_information',
        'family_information',
        'school_information',
        'career_information',
        'file_information',
        'discount_information',
        'payment_information',
        'sworn_declaration',
        'finish_registration'
    ];

    protected $appends = ['total_amount_pending'];

    // Columnas donde se puede buscar (texto general)
    public array $searchable = [];

    // Columnas donde se puede buscar (en relaciones)
    public array $searchableRelations = [
        'student' => [
            'document_number',
            'names',
            'paternal_surname',
            'maternal_surname',
            'email'
        ],
        'career' => ['name', 'code'],
        'cycle' => ['name', 'year'],
    ];

    // Columnas que se pueden ordenar
    public array $sortable = ['id', 'created_at', 'updated_at', 'status'];

    // Relaciones que se pueden cargar
    public array $allowedIncludes = [
        'student',
        'student.school',
        'student.residenceDistrict',
        'career',
        'cycle',
        'classroom',
        'payments',
        'discountType'
    ];

    // Columnas por las que se puede filtrar específicamente
    public array $filterable = [
        'status',
        'career_id',
        'classroom_id',
        'payment_type',
    ];


    // Obtiene el monto pendiente: total - descuento - pagos aprobados (solo concepto "Enrollment")
    public function getTotalAmountPendingAttribute(): string
    {
        $total = $this->total_amount ?? 0;
        $discount = $this->discount_amount ?? 0;
        // $paid = $this->payments()->approvedEnrollment()->sum('amount') ?? 0;
        $paid = $this->payments()->pendingEnrollment()->sum('amount') ?? 0;

        return number_format(max($total - $discount - $paid, 0), 2, '.', '');
    }

    // Agregar un paso
    public function addStep(string $step): bool
    {
        $steps = $this->registration_steps ?? [];

        // Evitar duplicados
        if (!in_array($step, $steps)) {
            $steps[] = $step;
            $this->registration_steps = $steps;
            return $this->save();
        }

        return true;
    }

    // Remover un paso
    public function removeStep(string $step): bool
    {
        $steps = $this->registration_steps ?? [];

        // Remover el paso si existe
        $steps = array_values(array_filter($steps, function ($s) use ($step) {
            return $s !== $step;
        }));

        $this->registration_steps = $steps;
        return $this->save();
    }

    // Verificar si un paso fue completado
    public function hasCompletedStep(string $step): bool
    {
        $steps = $this->registration_steps ?? [];
        return in_array($step, $steps);
    }

    // Obtener pasos completados
    public function getCompletedSteps(): array
    {
        return $this->registration_steps ?? [];
    }

    // Verificar si todos los pasos están completados
    public function isCompleted(): bool
    {
        $completedSteps = $this->getCompletedSteps();

        foreach (self::REQUIRED_STEPS as $step) {
            if (!in_array($step, $completedSteps)) {
                return false;
            }
        }

        return true;
    }

    // Obtener pasos faltantes
    public function getPendingSteps(): array
    {
        $completedSteps = $this->getCompletedSteps();

        return array_values(array_filter(self::REQUIRED_STEPS, function ($step) use ($completedSteps) {
            return !in_array($step, $completedSteps);
        }));
    }

    // Scope para matrículas activas
    public function scopeActive($query)
    {
        return $query->where('active', true);
    }

    // Relaciones
    public function student()
    {
        return $this->belongsTo(Student::class);
    }

    public function cycle()
    {
        return $this->belongsTo(Cycle::class);
    }

    public function career()
    {
        return $this->belongsTo(Career::class);
    }

    public function classroom()
    {
        return $this->belongsTo(Classroom::class);
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    public function discountType()
    {
        return $this->belongsTo(DiscountType::class);
    }
}
