<?php

namespace App\Models;

use App\Traits\HasUuidAndSlug;
use App\Traits\QueryScopeTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Cycle extends Model
{
    use HasFactory, QueryScopeTrait, HasUuidAndSlug;

    protected $fillable = [
        'name',
        'year',
        'enrollment_open',
        'active',
        'enrollment_start_date',
        'enrollment_end_date',
        'full_payment_cost',
        'installment_cost',
    ];

    protected $casts = [
        'active' => 'boolean',
        'enrollment_open' => 'boolean',
        'enrollment_start_date' => 'date',
        'enrollment_end_date' => 'date',
        'full_payment_cost' => 'decimal:2',
        'installment_cost' => 'decimal:2',
    ];

    // Columnas donde se puede buscar (texto general)
    public array $searchable = ['name', 'year'];

    // Columnas por las que se puede ordenar
    public array $sortable = ['id', 'name', 'year'];

    // Relaciones que se pueden incluir
    public array $allowedIncludes = ['enrollments'];

    // Columnas por las que se puede filtrar específicamente
    public array $filterable = ['id', 'name', 'year'];

    // HasUuidAndSlug trait: Slug
    protected function sluggableString(): string
    {
        return "{$this->name}-{$this->year}";
    }
    // HasUuidAndSlug trait: Slug
    protected function getSlugSourceFields(): array
    {
        return ['name', 'year'];
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('active', true);
    }

    public static function getLastActive(): ?self
    {
        return self::active()->latest('id')->first();
    }

    // Relaciones
    public function enrollments()
    {
        return $this->hasMany(Enrollment::class);
    }
}
