<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use App\Enums\Gender;

class TeacherRequest extends FormRequest
{
    // Autorizar todas las peticiones (validación de permisos en middleware)
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        switch ($this->route()->getActionMethod()) {
            case 'login':
                return $this->loginTeacherRules();
            case 'store':
                return $this->storeTeacherRules();
            case 'update':
                return $this->updateTeacherRules();
            default:
                return [];
        }
    }

    private function loginTeacherRules(): array
    {
        return [
            'document_number' => ['required', 'string', 'min:8', 'max:20'],
            'password' => ['required', 'string', 'min:8', 'max:50'],
        ];
    }

    private function storeTeacherRules(): array
    {
        return [
            'document_number' => ['required', 'string', 'digits:8', Rule::unique('teachers', 'document_number')],
            'names' => ['required', 'string', 'max:50'],
            'paternal_surname' => ['required', 'string', 'max:50'],
            'maternal_surname' => ['required', 'string', 'max:50'],
            'gender' => ['required', Rule::enum(Gender::class)],
            'birth_date' => ['required', 'date', 'before:today'],
            'email' => ['required', 'email', Rule::unique('teachers', 'email')],
            'phone_number' => ['required', 'string', 'digits:9'],
            'photo_teacher' => ['nullable', 'string', 'max:255'],
        ];
    }

    private function updateTeacherRules(): array
    {
        return [
            'document_number' => ['required', 'string', 'digits:8', Rule::unique('teachers', 'document_number')->ignore($this->route('id'))],
            'names' => ['required', 'string', 'max:50'],
            'paternal_surname' => ['required', 'string', 'max:50'],
            'maternal_surname' => ['required', 'string', 'max:50'],
            'gender' => ['required', Rule::enum(Gender::class)],
            'birth_date' => ['required', 'date', 'before:today'],
            'email' => ['required', 'email'],
            'phone_number' => ['required', 'string', 'digits:9'],
            'photo_teacher' => ['nullable', 'string', 'max:255'],
        ];
    }

    // Reglas de validación
    // public function rules(): array
    // {
    //     $teacherId = $this->route('id');
    //     return [
    //         'document_number' => [
    //             'required',
    //             'string',
    //             'digits:8',
    //             Rule::unique('teachers', 'document_number')->ignore($teacherId)
    //         ],
    //         'names' => ['required', 'string', 'max:50'],
    //         'paternal_surname' => ['required', 'string', 'max:50'],
    //         'maternal_surname' => ['required', 'string', 'max:50'],
    //         'gender' => ['required', Rule::enum(Gender::class)],
    //         'birth_date' => ['required', 'date', 'before:today'],
    //         'email' => [
    //             'required',
    //             'email',
    //             Rule::unique('teachers', 'email')->ignore($teacherId),
    //         ],
    //         'phone_number' => ['required', 'string', 'digits:9'],
    //         'photo_teacher' => ['nullable', 'string', 'max:255'],
    //     ];
    // }

    // Etiquetas de validación
    public function attributes(): array
    {
        return [
            'document_number' => 'Documento',
            'names' => 'Nombres',
            'paternal_surname' => 'Apellido paterno',
            'maternal_surname' => 'Apellido materno',
            'gender' => 'Genero',
            'birth_date' => 'Fecha de nacimiento',
            'email' => 'Correo',
            'phone_number' => 'Teléfono',
            'photo_teacher' => 'Foto',
        ];
    }
}
