<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class EnrollmentRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        // Detecta qué función del controlador se está ejecutando
        switch ($this->route()->getActionMethod()) {
            case 'registerPersonalInformation':
                return $this->personalDataRules();
            case 'registerAdditionalInformation':
                return $this->additionalDataRules();
            case 'registerFamilyInformation':
                return $this->familyDataRules();
            case 'registerSchoolInformation':
                return $this->schoolDataRules();
            case 'registerCareerInformation':
                return $this->careerDataRules();
            case 'uploadDocument':
                return $this->uploadDocumentRules();
            case 'swornDeclaration':
                return $this->swornDeclarationRules();
            default:
                return [];
        }
    }

    private function personalDataRules(): array
    {
        return [
            'document_type' => ['required', 'in:dni,pasaporte,carnet_extranjeria'],
            'document_number' => ['required', 'string', 'min:8', 'max:20'],
            'names' => ['required', 'string', 'min:2', 'max:255'],
            'paternal_surname' => ['required', 'string', 'min:2', 'max:255'],
            'maternal_surname' => ['required', 'string', 'min:2', 'max:255'],
            'birth_date' => ['required', 'date', 'before:today'],
            'gender' => ['required', 'in:masculino,femenino,otro'],
            'phone_number' => ['required', 'string', 'size:9'],
            'whatsapp_number' => ['required', 'string', 'size:9'],
            'email' => ['required', 'email', 'max:255'],
        ];
    }

    private function additionalDataRules(): array
    {
        return [
            'birth_place_id' => ['required', 'exists:districts,id'],
            'marital_status' => ['required', 'in:soltero,casado,divorciado,viudo'],
            'languages' => ['required', 'array'],
            'residence_address' => ['required', 'string', 'min:2', 'max:255'],
            'residence_place_id' => ['required', 'exists:districts,id'],
        ];
    }

    private function familyDataRules(): array
    {
        return [
            'family_data' => ['required', 'array'],
        ];
    }

    private function schoolDataRules(): array
    {
        $rules = [
            'school_id' => ['required', 'exists:schools,id'],
        ];

        if ($this->filled('school_graduation_year')) {
            $rules['school_graduation_year'] = ['digits:4', 'integer', 'min:1900', 'max:' . now()->year];
        } else {
            $rules['grade_school'] = ['required', 'in:primero,segundo,tercero,cuarto,quinto'];
        }

        return $rules;
    }

    private function careerDataRules(): array
    {
        return [
            'career_id' => ['required', 'exists:careers,id'],
            'information_sources' => ['required', 'array'],
        ];
    }

    private function uploadDocumentRules(): array
    {
        return [
            'type' => ['required', 'in:document_file,photo_student,file_school'],
            'file' => ['required', 'file', 'mimes:pdf,jpg,jpeg,png'],
        ];
    }

    private function swornDeclarationRules(): array
    {
        return [
            'sworn_declaration' => ['required', 'boolean'],
        ];
    }

    public function attributes(): array
    {
        return [
            'document_type' => 'Tipo de Documento',
            'document_number' => 'Número de Documento',
            'names' => 'Nombres',
            'paternal_surname' => 'Apellido Paterno',
            'maternal_surname' => 'Apellido Materno',
            'birth_date' => 'Fecha de Nacimiento',
            'gender' => 'Género',
            'phone_number' => 'Teléfono',
            'whatsapp_number' => 'WhatsApp',
            'email' => 'Email',
            'birth_place_id' => 'Lugar de Nacimiento',
            'marital_status' => 'Estado Civil',
            'languages' => 'Idiomas',
            'residence_address' => 'Dirección',
            'residence_place_id' => 'Lugar de Residencia',
            'school_id' => 'Colegio',
            'school_graduation_year' => 'Año de Egreso',
            'grade_school' => 'Grado',
            'family_data' => 'Datos Familiares',
            'career_id' => 'Carrera',
            'information_sources' => 'Medios de Información',
            'type' => 'Tipo',
            'file' => 'Archivo',
            'sworn_declaration' => 'Declaración Jurada',
        ];
    }
}
