<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

abstract class BaseAuth
{
    protected string $model;
    protected string $guardName;

    public function handle(Request $request, Closure $next): Response
    {
        // Obtener token del header Authorization
        $token = $request->bearerToken();

        if (!$token) {
            return $this->unauthorizedResponse('Token de autenticación requerido');
        }

        // Buscar usuario por token hasheado
        $user = $this->findUserByToken($token);

        if (!$user) {
            return $this->unauthorizedResponse('Token inválido o expirado');
        }

        // Actualizar timestamp de último acceso
        $user->updateLastLogin();

        // Inyectar usuario autenticado en el request
        $request->setUserResolver(fn() => $user);

        return $next($request);
    }

    // Buscar usuario por token en BD usando hash SHA-256
    private function findUserByToken(string $plainToken)
    {
        $hashedToken = hash('sha256', $plainToken);

        // Forma correcta de instanciar un modelo desde un string
        $modelClass = $this->model;

        return $modelClass::where('access_token', $hashedToken)
            ->where('token_expires_at', '>', now())
            ->first();

        // return ($this->model)::where('access_token', $hashedToken)
        //     ->where('token_expires_at', '>', now())
        //     ->first();
    }

    // Respuesta estandarizada para errores de autenticación
    private function unauthorizedResponse(string $message): Response
    {
        return response()->json([
            'success' => false,
            'message' => $message,
        ], 401);
    }
}
