<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\RoleResource;
use App\Models\Student;
use App\Models\Teacher;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class UserRoleController extends Controller
{

    // Obtener todos los roles de un usuario/teacher/student
    public function index(string $modelType, int $modelId): JsonResponse
    {
        $model = $this->getModel($modelType, $modelId);

        $roles = $model->roles;

        return response()->json([
            'success' => true,
            'message' => 'Roles obtenidos exitosamente',
            'data' => RoleResource::collection($roles),
        ]);
    }

    // Asignar uno o más roles a un usuario/teacher/student (sin eliminar los existentes)
    public function attach(Request $request, string $modelType, int $modelId): JsonResponse
    {
        // Validar que se envíen slugs de roles válidos
        $validated = $request->validate([
            'role_slugs' => 'required|array|min:1',
            'role_slugs.*' => 'required|string|exists:roles,slug',
        ], [
            'role_slugs.required' => 'Debe enviar al menos un rol',
            'role_slugs.*.exists' => 'Uno o más roles no existen',
        ]);

        $model = $this->getModel($modelType, $modelId);

        try {
            // Usar el método del trait HasRoles
            $model->assignRole($validated['role_slugs']);
        } catch (\Exception $e) {
            throw ValidationException::withMessages([
                'role_slugs' => $e->getMessage(),
            ]);
        }

        // Recargar roles actualizados
        $model->load('roles');

        return response()->json([
            'success' => true,
            'message' => 'Roles asignados exitosamente',
            'data' => RoleResource::collection($model->roles),
        ]);
    }

    // Remover uno o más roles de un usuario/teacher/student
    public function detach(Request $request, string $modelType, int $modelId): JsonResponse
    {
        // Validar que se envíen slugs de roles válidos
        $validated = $request->validate([
            'role_slugs' => 'required|array|min:1',
            'role_slugs.*' => 'required|string',
        ], [
            'role_slugs.required' => 'Debe enviar al menos un rol',
        ]);

        $model = $this->getModel($modelType, $modelId);

        // Usar el método del trait HasRoles
        $model->removeRole($validated['role_slugs']);

        // Recargar roles actualizados
        $model->load('roles');

        return response()->json([
            'success' => true,
            'message' => 'Roles removidos exitosamente',
            'data' => RoleResource::collection($model->roles),
        ]);
    }

    // Sincronizar roles (reemplaza todos los roles actuales)
    public function sync(Request $request, string $modelType, int $modelId): JsonResponse
    {
        // Validar que se envíen slugs de roles (puede ser array vacío para remover todos)
        $validated = $request->validate([
            'role_slugs' => 'required|array',
            'role_slugs.*' => 'required|string|exists:roles,slug',
        ], [
            'role_slugs.required' => 'Debe enviar un array de roles',
            'role_slugs.*.exists' => 'Uno o más roles no existen',
        ]);

        $model = $this->getModel($modelType, $modelId);

        // Si se envía un array vacío, se eliminan todos los roles
        if (empty($validated['role_slugs'])) {
            $model->removeAllRoles();
            return response()->json([
                'success' => true,
                'message' => 'Todos los roles fueron removidos',
                'data' => [],
            ]);
        }

        try {
            // Usar el método del trait HasRoles
            $model->syncRoles($validated['role_slugs']);
        } catch (\Exception $e) {
            throw ValidationException::withMessages([
                'role_slugs' => $e->getMessage(),
            ]);
        }

        // Recargar roles actualizados
        $model->load('roles');

        return response()->json([
            'success' => true,
            'message' => 'Roles sincronizados exitosamente',
            'data' => RoleResource::collection($model->roles),
        ]);
    }

    // Helper privado para obtener el modelo correcto según el tipo
    private function getModel(string $modelType, int $modelId)
    {
        // Validar que el tipo de modelo sea válido
        $modelClass = match ($modelType) {
            'users' => User::class,
            'teachers' => Teacher::class,
            'students' => Student::class,
            default => throw ValidationException::withMessages([
                'model_type' => 'Tipo de modelo inválido. Use: users, teachers o students',
            ]),
        };

        // // Buscar el modelo por ID
        // return $this->findOrFail(
        //     $modelClass,
        //     'id',
        //     $modelId,
        //     ucfirst($modelType) . ' no encontrado.'
        // );

        $item = $modelClass::findOrAbort($modelId, ucfirst($modelType) . ' no encontrado.');
    }
}
