<?php

namespace App\Http\Controllers\Api\Student\Enrollment;

use App\Enums\EnrollmentStatus;
use App\Enums\PaymentStatus;
use App\Enums\PaymentType;
use App\Http\Controllers\Controller;
use App\Http\Requests\Enrollment\AdditionalDataRequest;
use App\Http\Requests\Enrollment\CareerDataRequest;
use App\Http\Requests\Enrollment\FamilyDataRequest;
use App\Http\Requests\Enrollment\FileUploadRequest;
use App\Http\Requests\Enrollment\PaymentDataRequest;
use App\Http\Requests\Enrollment\PersonalDataRequest;
use App\Http\Requests\Enrollment\SchoolDataRequest;
use App\Http\Requests\Enrollment\SwornDeclarationRequest;
use App\Http\Resources\EnrollmentResource;
use App\Http\Resources\StudentResource;
use App\Models\Payment;
use App\Rules\FileExtension;
use App\Services\EnrollmentService;
use App\Services\FileStorageService;
use App\Traits\AppHelpers;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class EnrollmentController extends Controller
{
    use AppHelpers;

    private EnrollmentService $enrollmentService;
    private FileStorageService $fileStorageService;

    public function __construct(
        EnrollmentService $enrollmentService,
        FileStorageService $fileStorageService
    ) {
        $this->enrollmentService = $enrollmentService;
        $this->fileStorageService = $fileStorageService;
    }
    // Paso 1: Actualizar información personal
    public function registerPersonalInformation(PersonalDataRequest $request): JsonResponse
    {
        $validated = $request->validated();
        $student = $request->user();
        $activeEnrollment = $this->getActiveEnrollment($student);

        $student->update($validated);
        $activeEnrollment->addStep('personal_data');

        return $this->buildEnrollmentResponse(
            $student->fresh(),
            $activeEnrollment->fresh(),
            'Información personal actualizada exitosamente'
        );
    }

    // Paso 2: Actualizar información adicional
    public function registerAdditionalInformation(AdditionalDataRequest $request): JsonResponse
    {
        $validated = $request->validated();
        $student = $request->user();
        $activeEnrollment = $this->getActiveEnrollment($student);

        $student->update($validated);
        $activeEnrollment->addStep('additional_information');

        return $this->buildEnrollmentResponse(
            $student->fresh(),
            $activeEnrollment->fresh(),
            'Información adicional actualizada exitosamente'
        );
    }


    // Paso 3: Actualizar información familiar
    public function registerFamilyInformation(FamilyDataRequest $request): JsonResponse
    {
        $validated = $request->validated();
        $student = $request->user();
        $activeEnrollment = $this->getActiveEnrollment($student);

        $student->update($validated);
        $activeEnrollment->addStep('family_information');

        return $this->buildEnrollmentResponse(
            $student->fresh(),
            $activeEnrollment->fresh(),
            'Información de familia actualizada exitosamente'
        );
    }


    // Paso 4: Actualizar información de colegio
    public function registerSchoolInformation(SchoolDataRequest $request): JsonResponse
    {
        $validated = $request->validated();
        $student = $request->user();
        $activeEnrollment = $this->getActiveEnrollment($student);

        $student->update(['school_id' => $validated['school_id']]);

        if (isset($validated['school_graduation_year'])) {
            $student->update([
                'school_graduation_year' => $validated['school_graduation_year'],
                'school_graduation_certificate' => null,
            ]);
            $activeEnrollment->update([
                'grade_school' => null,
                'file_grade_school' => null,
            ]);
        } else {
            $activeEnrollment->update([
                'grade_school' => $validated['grade_school'],
                'file_grade_school' => null,
            ]);
            $student->update([
                'school_graduation_year' => null,
                'school_graduation_certificate' => null,
            ]);
        }

        $activeEnrollment->addStep('school_information');
        $activeEnrollment->removeStep('file_information');

        return $this->buildEnrollmentResponse(
            $student->fresh(),
            $activeEnrollment->fresh(),
            'Información de colegio actualizada exitosamente'
        );
    }

    // Paso 5: Actualizar información de carrera
    public function registerCareerInformation(CareerDataRequest $request): JsonResponse
    {
        $validated = $request->validated();
        $student = $request->user();
        $activeEnrollment = $this->getActiveEnrollment($student);

        $activeEnrollment->update($validated);
        $activeEnrollment->addStep('career_information');

        return $this->buildEnrollmentResponse(
            $student->fresh(),
            $activeEnrollment->fresh(),
            'Información de carrera actualizada exitosamente'
        );
    }

    // Paso 6: Subir Documentos
    public function uploadDocument(FileUploadRequest $request): JsonResponse
    {
        $validated = $request->validated();
        $student = $request->user();
        $activeEnrollment = $this->getActiveEnrollment($student);

        $config = $this->getFileConfig($validated['type'], $student, $activeEnrollment);

        if (!$config) {
            return response()->json([
                'success' => false,
                'message' => 'Tipo de documento inválido',
            ], 400);
        }

        $result = $this->fileStorageService->update(
            file: $request->file('file'),
            disk: 'local',
            path: $config['path'],
            oldFilePath: $config['old_file'] ?? '',
            allowedTypes: $config['allowed_types'],
            maxSizeKB: 5120,
        );

        if (!$result['success']) {
            return response()->json($result, 400);
        }

        $this->updateFileRecords($config, $result['data']['path'], $student, $activeEnrollment);

        $student->refresh();
        $activeEnrollment->refresh();

        if ($this->hasAllRequiredDocuments($student, $activeEnrollment)) {
            $activeEnrollment->addStep('file_information');
        }

        return $this->buildEnrollmentResponse(
            $student->fresh(),
            $activeEnrollment->fresh(),
            'Documento actualizado exitosamente'
        );
    }

    // Paso 7: Aceptar Declaración Jurada
    public function swornDeclaration(SwornDeclarationRequest $request): JsonResponse
    {
        $validated = $request->validated();
        $student = $request->user();
        $activeEnrollment = $this->getActiveEnrollment($student);

        $activeEnrollment->update(['oath_declaration' => $validated['sworn_declaration']]);

        if ($validated['sworn_declaration']) {
            $activeEnrollment->addStep('sworn_declaration');
            $message = 'Declaración jurada aceptada exitosamente';
        } else {
            $activeEnrollment->removeStep('sworn_declaration');
            $message = 'Debe aceptar la declaración jurada para continuar';
        }

        return $this->buildEnrollmentResponse(
            $student->fresh(),
            $activeEnrollment->fresh(),
            $message
        );
    }

    // Paso 8: Registrar Descuento
    public function registerDiscount(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'payment_type' => ['required', 'string', Rule::in(PaymentType::values())],
            'discount_type_id' => ['required', 'exists:discount_types,id'],
            'discount_amount' => ['required', 'numeric', 'min:0'],
            'discount_file' => ['nullable', 'file', 'max:5000', new FileExtension(['pdf', 'jpg', 'jpeg', 'png'])],
        ]);

        $student = $request->user();
        $activeEnrollment = $this->getActiveEnrollment($student);
        $cycle = $this->getActiveCycle();

        // Sacar monto 
        if ($validated['payment_type'] == PaymentType::Full->value) {
            $validated['total_amount'] = $cycle->full_payment_cost;
        } else if ($validated['payment_type'] == PaymentType::Installment->value) {
            $validated['total_amount'] = $cycle->installment_cost;
        } else {
            $validated['total_amount'] = 0;
        }

        // Subir Documento
        if ($request->hasFile('discount_file')) {
            $result = $this->fileStorageService->upload(
                file: $request->file('discount_file'),
                disk: 'local',
                path: "students/{$student->id}/cycle/{$activeEnrollment->cycle_id}/discounts/",
                allowedTypes: ['pdf', 'jpg', 'jpeg', 'png'],
                maxSizeKB: 5120,
            );
        }

        # Actualizar Descuento
        $activeEnrollment->update([
            'total_amount' => $validated['total_amount'],
            'payment_type' => $validated['payment_type'],
            'discount_type_id' => $validated['discount_type_id'],
            'discount_amount' => $validated['discount_amount'],
            'discount_file' => $result['data']['path'] ?? null
        ]);

        $activeEnrollment->addStep('discount_information');

        return $this->buildEnrollmentResponse(
            $student->fresh(),
            $activeEnrollment->fresh(),
            'Tipo de Pago actualizado exitosamente'
        );
    }


    // Paso 9: Registrar Pago
    public function registerPayment(PaymentDataRequest $request): JsonResponse
    {
        $validated = $request->validated();
        $student = $request->user();
        $activeEnrollment = $this->getActiveEnrollment($student);

        // Subir Documento
        $result = $this->fileStorageService->upload(
            file: $request->file('file'),
            disk: 'local',
            path: "students/{$student->id}/cycle/{$activeEnrollment->cycle_id}/payments/",
            allowedTypes: ['png', 'jpg', 'jpeg'],
            maxSizeKB: 5120,
        );

        if (!$result['success']) {
            return response()->json($result, 400);
        }

        // Registrar Pago
        $payment = $activeEnrollment->payments()->create([
            'amount' => $validated['amount'],
            'method' => 'box',
            'payment_date' => $validated['payment_date'],
            'receipt_number' => $validated['receipt_number'],
            'document_number' => $validated['document_number'] ?? null,
            'full_name' => $validated['full_name'] ?? null,
            'comment' => $validated['comment'] ?? null,
            'file' => $result['data']['path'],
            'status' => PaymentStatus::Pending->value
        ]);

        $activeEnrollment->addStep('payment_information');

        return $this->buildEnrollmentResponse(
            $student->fresh(),
            $activeEnrollment->fresh(),
            'Pago registrado exitosamente'
        );
    }

    // Eliminar Pago
    public function deletePayment(Request $request, $payment_id): JsonResponse
    {
        $payment = Payment::findOrAbort($payment_id, 'Pago no encontrado');

        if ($payment->status != PaymentStatus::Pending) {
            return response()->json([
                'success' => false,
                'message' => 'No se puede eliminar este Pago',
                'data' => $payment,
            ], 400);
        }

        $student = $request->user();
        $activeEnrollment = $this->getActiveEnrollment($student);

        try {
            DB::beginTransaction();

            $filePath = $payment->file;

            $payment->delete();

            DB::commit();

            if ($filePath) {
                $this->fileStorageService->delete(
                    path: $filePath,
                    disk: 'local'
                );
            }

            return $this->buildEnrollmentResponse(
                $student->fresh(),
                $activeEnrollment->fresh(),
                'Pago eliminado exitosamente'
            );
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'No se pudo eliminar el pago: ' . $e->getMessage(),
                'data' => null,
            ], 400);
        }
    }

    // Paso 10: Finalizar Registro
    public function finishRegistration(Request $request): JsonResponse
    {
        $student = $request->user();
        $activeEnrollment = $this->getActiveEnrollment($student);

        $activeEnrollment->update(['status' => EnrollmentStatus::Finished->value]);

        $activeEnrollment->addStep('finish_registration');

        return $this->buildEnrollmentResponse(
            $student->fresh(),
            $activeEnrollment->fresh(),
            'Registro finalizado exitosamente'
        );
    }


    // ========================================
    // MÉTODOS PRIVADOS
    // ========================================

    /**
     * Construye respuesta estandarizada para enrollment
     */
    private function buildEnrollmentResponse($student, $enrollment, string $message): JsonResponse
    {
        // Cargar relaciones necesarias
        $this->loadStudentRelations($student);
        $enrollment->load('payments');

        // Obtener token
        $token = $student->plain_token ?? $student->getPlainTokenFromRequest();

        // Usar helper para respuesta estandarizada
        return auth_response_success(
            user: StudentResource::make($student)->resolve(),
            token: $token,
            tokenExpires: $student->token_expires_at->toIso8601String(),
            meta: [
                'enrollment' => EnrollmentResource::make($enrollment)->resolve(),
                'registration_steps' => $enrollment->getCompletedSteps(),
                'is_registration_completed' => $enrollment->isCompleted(),
            ],
            message: $message
        );
    }

    /**
     * Cargar relaciones del estudiante de forma optimizada
     */
    private function loadStudentRelations($student): void
    {
        $student->load([
            'school:id,name,code,sector,level,ugel,validated,district_id',
            'school.district:id,name,province_id',
            'school.district.province:id,name,department_id',
            'school.district.province.department:id,name,country_id',
            'school.district.province.department.country:id,name',
            'residenceDistrict:id,name,province_id',
            'residenceDistrict.province:id,name,department_id',
            'residenceDistrict.province.department:id,name,country_id',
            'residenceDistrict.province.department.country:id,name',
            'birthDistrict:id,name,province_id',
            'birthDistrict.province:id,name,department_id',
            'birthDistrict.province.department:id,name,country_id',
            'birthDistrict.province.department.country:id,name'
        ]);
    }

    // Obtiene configuración según tipo de archivo
    private function getFileConfig(string $type, $student, $enrollment): ?array
    {
        $isGraduated = !empty($student->school_graduation_year);

        $configs = [
            'document_file' => [
                'path' => "students/{$student->id}/document_file/",
                'old_file' => $student->document_file,
                'allowed_types' => ['pdf'],
                'updates' => [
                    'student' => ['document_file' => 'path'],
                ],
            ],
            'photo_student' => [
                'path' => "students/{$student->id}/photo_student/",
                'old_file' => $student->photo_student,
                'allowed_types' => ['jpg', 'jpeg', 'png'],
                'updates' => [
                    'student' => ['photo_student' => 'path'],
                ],
            ],
            'file_school' => [
                'path' => $isGraduated
                    ? "students/{$student->id}/school_graduation_certificate/"
                    : "students/{$student->id}/cycle/{$enrollment->cycle_id}/file_grade_school/",
                'old_file' => $isGraduated
                    ? $student->school_graduation_certificate
                    : $enrollment->file_grade_school,
                'allowed_types' => ['pdf'],
                'updates' => $isGraduated ? [
                    'student' => ['school_graduation_certificate' => 'path'],
                    'enrollment' => ['file_grade_school' => null],
                ] : [
                    'student' => ['school_graduation_certificate' => null],
                    'enrollment' => ['file_grade_school' => 'path'],
                ],
            ],
        ];

        return $configs[$type] ?? null;
    }

    // Actualiza registros en BD según configuración
    private function updateFileRecords(array $config, string $filePath, $student, $enrollment): void
    {
        foreach ($config['updates'] as $model => $fields) {
            $updates = [];

            foreach ($fields as $field => $value) {
                $updates[$field] = $value === 'path' ? $filePath : $value;
            }

            if ($model === 'student') {
                $student->update($updates);
            } elseif ($model === 'enrollment') {
                $enrollment->update($updates);
            }
        }
    }

    // Valida que existan los 3 archivos obligatorios
    private function hasAllRequiredDocuments($student, $enrollment): bool
    {
        return !empty($student->document_file) &&
            !empty($student->photo_student) &&
            (!empty($student->school_graduation_certificate) || !empty($enrollment->file_grade_school));
    }
}
