<?php

namespace App\Http\Controllers\Api\Shared;

use App\Http\Controllers\Controller;
use App\Services\FileStorageService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\StreamedResponse;

class FileStorageController extends Controller
{
    private FileStorageService $fileStorageService;

    public function __construct(FileStorageService $fileStorageService)
    {
        $this->fileStorageService = $fileStorageService;
    }

    // Sube un archivo genérico o asociado a un modelo
    public function uploadFile(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'file' => 'required|file',
            'disk' => 'required|string|in:public,local',
            'path' => 'required|string',
            'model' => 'nullable|string',
            'model_id' => 'nullable|integer',
            'field' => 'nullable|string',
            'allowed_types' => 'nullable|array',
            'max_size' => 'nullable|integer',
        ]);

        // Subir archivo usando el service
        $result = $this->fileStorageService->upload(
            file: $request->file('file'),
            disk: $validated['disk'],
            path: $validated['path'],
            allowedTypes: $validated['allowed_types'] ?? null,
            maxSize: $validated['max_size'] ?? null,
        );

        if (!$result['success']) {
            return response()->json($result, 400);
        }

        // Si hay modelo, actualizar registro en BD
        if (!empty($validated['model']) && !empty($validated['model_id']) && !empty($validated['field'])) {
            $modelClass = "App\\Models\\{$validated['model']}";
            $record = $modelClass::findOrFail($validated['model_id']);

            $record->update([
                $validated['field'] => $result['data']['path']
            ]);

            $result['data']['updated_record'] = [
                'model' => $validated['model'],
                'model_id' => $validated['model_id'],
                'field' => $validated['field'],
            ];
        }

        return response()->json($result, 201);
    }

    // Actualiza un archivo eliminando el anterior y subiendo uno nuevo
    public function updateFile(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'file' => 'required|file',
            'disk' => 'required|string|in:public,local',
            'path' => 'required|string',
            'old_file_path' => 'required|string',
            'model' => 'nullable|string',
            'model_id' => 'nullable|integer',
            'field' => 'nullable|string',
            'allowed_types' => 'nullable|array',
            'max_size' => 'nullable|integer',
        ]);

        // Actualizar archivo usando el service
        $result = $this->fileStorageService->update(
            file: $request->file('file'),
            disk: $validated['disk'],
            path: $validated['path'],
            oldFilePath: $validated['old_file_path'],
            allowedTypes: $validated['allowed_types'] ?? null,
            maxSize: $validated['max_size'] ?? null,
        );

        if (!$result['success']) {
            return response()->json($result, 400);
        }

        // Si hay modelo, actualizar registro en BD
        if (!empty($validated['model']) && !empty($validated['model_id']) && !empty($validated['field'])) {
            $modelClass = "App\\Models\\{$validated['model']}";
            $record = $modelClass::findOrFail($validated['model_id']);

            $record->update([
                $validated['field'] => $result['data']['path']
            ]);

            $result['data']['updated_record'] = [
                'model' => $validated['model'],
                'model_id' => $validated['model_id'],
                'field' => $validated['field'],
            ];
        }

        return response()->json($result, 200);
    }

    // Elimina un archivo del storage
    public function deleteFile(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'path' => 'required|string',
            'disk' => 'required|string|in:public,local',
            'model' => 'nullable|string',
            'model_id' => 'nullable|integer',
            'field' => 'nullable|string',
        ]);

        // Eliminar archivo usando el service
        $result = $this->fileStorageService->delete(
            path: $validated['path'],
            disk: $validated['disk']
        );

        if (!$result['success']) {
            return response()->json($result, 400);
        }

        // Si hay modelo, limpiar campo en BD
        if (!empty($validated['model']) && !empty($validated['model_id']) && !empty($validated['field'])) {
            $modelClass = "App\\Models\\{$validated['model']}";
            $record = $modelClass::findOrFail($validated['model_id']);

            $record->update([
                $validated['field'] => null
            ]);

            $result['data']['updated_record'] = [
                'model' => $validated['model'],
                'model_id' => $validated['model_id'],
                'field' => $validated['field'],
                'value' => null,
            ];
        }

        return response()->json($result, 200);
    }

    // Muestra un archivo en el navegador sin descargarlo
    public function displayFile(Request $request): StreamedResponse|JsonResponse
    {
        $validated = $request->validate([
            'path' => 'required|string',
            'disk' => 'required|string|in:public,local',
        ]);

        // Verificar si el archivo existe
        if (!$this->fileStorageService->exists($validated['path'], $validated['disk'])) {
            return response()->json([
                'success' => false,
                'message' => 'Archivo no encontrado',
                'data' => null,
            ], 404);
        }

        // Mostrar archivo usando el service
        return $this->fileStorageService->display(
            path: $validated['path'],
            disk: $validated['disk']
        );
    }

    // Descarga un archivo del storage
    public function downloadFile(Request $request): StreamedResponse|JsonResponse
    {
        $validated = $request->validate([
            'path' => 'required|string',
            'disk' => 'required|string|in:public,local',
        ]);

        // Verificar si el archivo existe
        if (!$this->fileStorageService->exists($validated['path'], $validated['disk'])) {
            return response()->json([
                'success' => false,
                'message' => 'Archivo no encontrado',
                'data' => null,
            ], 404);
        }

        // Descargar archivo usando el service
        return $this->fileStorageService->download(
            path: $validated['path'],
            disk: $validated['disk']
        );
    }

    // Verifica si un archivo existe en el storage
    public function checkFile(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'path' => 'required|string',
            'disk' => 'required|string|in:public,local',
        ]);

        // Verificar existencia usando el service
        $exists = $this->fileStorageService->exists(
            path: $validated['path'],
            disk: $validated['disk']
        );

        return response()->json([
            'success' => true,
            'message' => $exists ? 'Archivo encontrado' : 'Archivo no encontrado',
            'data' => [
                'exists' => $exists,
                'path' => $validated['path'],
                'disk' => $validated['disk'],
            ],
        ], 200);
    }

    public function getTemporaryUrl(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'path' => 'required|string',
            'disk' => 'required|string|in:public,local',
            'minutes' => 'nullable|integer|min:1|max:1440',
        ]);

        try {
            // Asegurar que sea integer
            $minutes = (int) ($validated['minutes'] ?? 2); // Por defecto 2 minutos

            $url = $this->fileStorageService->getTemporaryUrl(
                path: $validated['path'],
                disk: $validated['disk'],
                minutes: $minutes
            );

            return response()->json([
                'success' => true,
                'message' => 'URL temporal generada',
                'data' => [
                    'url' => $url,
                    'expires_in_minutes' => $validated['minutes'] ?? 60,
                    'expires_at' => now()->addMinutes($validated['minutes'] ?? 60)->toIso8601String(),
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
                'data' => null,
            ], 400);
        }
    }

    // Muestra archivo usando URL firmada (NO requiere autenticación)
    public function displayFileSigned(Request $request): StreamedResponse|JsonResponse
    {
        // Validar firma
        if (!$request->hasValidSignature()) {
            return response()->json([
                'success' => false,
                'message' => 'URL expirada o inválida',
                'data' => null,
            ], 403);
        }

        $path = $request->query('path');
        $disk = $request->query('disk', 'local');

        if (!$this->fileStorageService->exists($path, $disk)) {
            return response()->json([
                'success' => false,
                'message' => 'Archivo no encontrado',
                'data' => null,
            ], 404);
        }

        return $this->fileStorageService->display($path, $disk);
    }
}
/*
|--------------------------------------------------------------------------
| DOCUMENTACIÓN DE USO - FileStorageController
|--------------------------------------------------------------------------
|
| Controller genérico para administrar archivos desde el frontend.
| Puedes usar este controller directamente o copiar las funciones a
| controllers específicos de cada modelo (StudentController, TeacherController, etc.)
|
|--------------------------------------------------------------------------
| RUTAS API (routes/api.php)
|--------------------------------------------------------------------------
|
| use App\Http\Controllers\Api\FileStorageController;
|
| Route::middleware('auth:sanctum')->prefix('files')->group(function () {
|     Route::post('/upload', [FileStorageController::class, 'uploadFile']);
|     Route::post('/update', [FileStorageController::class, 'updateFile']);
|     Route::delete('/delete', [FileStorageController::class, 'deleteFile']);
|     Route::get('/display', [FileStorageController::class, 'displayFile']);
|     Route::get('/download', [FileStorageController::class, 'downloadFile']);
|     Route::get('/check', [FileStorageController::class, 'checkFile']);
| });
|
|--------------------------------------------------------------------------
| FUNCIONES DISPONIBLES
|--------------------------------------------------------------------------
|
| 1. uploadFile() - Sube un archivo nuevo al storage
|    - Uso: Subir documentos, fotos, certificados por primera vez
|    - Puede asociarse a un modelo o ser genérico
|    - Endpoint: POST /api/files/upload
|
| 2. updateFile() - Actualiza un archivo existente
|    - Uso: Reemplazar un documento anterior con uno nuevo
|    - Elimina automáticamente el archivo anterior
|    - Endpoint: POST /api/files/update
|
| 3. deleteFile() - Elimina un archivo del storage
|    - Uso: Borrar documentos obsoletos o incorrectos
|    - Puede limpiar el campo en BD si se asocia a modelo
|    - Endpoint: DELETE /api/files/delete
|
| 4. displayFile() - Muestra un archivo en el navegador
|    - Uso: Ver PDFs, imágenes sin descargarlos
|    - Solo para archivos privados (disco: local)
|    - Endpoint: GET /api/files/display
|
| 5. downloadFile() - Descarga un archivo
|    - Uso: Permitir descargas de archivos privados
|    - Solo para archivos privados (disco: local)
|    - Endpoint: GET /api/files/download
|
| 6. checkFile() - Verifica si un archivo existe
|    - Uso: Validar existencia antes de mostrar/descargar
|    - Retorna booleano con estado del archivo
|    - Endpoint: GET /api/files/check
|
|--------------------------------------------------------------------------
| PARÁMETROS COMUNES
|--------------------------------------------------------------------------
|
| OBLIGATORIOS:
| - file: Archivo a subir (UploadedFile) - Solo en upload/update
| - disk: Disco de almacenamiento ('public' o 'local')
| - path: Ruta dentro del disco (ej: 'students/documents')
|
| OPCIONALES (para asociar con modelo):
| - model: Nombre del modelo (ej: 'Student', 'Teacher')
| - model_id: ID del registro en BD
| - field: Campo de la tabla donde guardar la ruta
|
| OPCIONALES (validaciones):
| - allowed_types: Array de extensiones permitidas (ej: ['pdf', 'jpg'])
| - max_size: Tamaño máximo en KB (ej: 5120 = 5MB)
|
| ESPECÍFICOS:
| - old_file_path: Ruta del archivo anterior (solo en update)
|
|--------------------------------------------------------------------------
| CASOS DE USO PRÁCTICOS
|--------------------------------------------------------------------------
|
| === CASO 1: Subir documento de estudiante ===
| 1. Frontend envía archivo a POST /api/files/upload
| 2. Incluye: model=Student, model_id=123, field=document_file
| 3. Backend guarda archivo y actualiza BD automáticamente
| 4. Retorna ruta del archivo guardado
|
| === CASO 2: Subir foto de perfil pública ===
| 1. Frontend envía archivo a POST /api/files/upload
| 2. Incluye: disk=public, path=students/photos
| 3. Backend guarda en storage/app/public/
| 4. Archivo accesible vía URL: /storage/students/photos/uuid_foto.jpg
|
| === CASO 3: Ver documento privado ===
| 1. Frontend solicita GET /api/files/display?path=...&disk=local
| 2. Backend verifica permisos (middleware auth:sanctum)
| 3. Retorna archivo para mostrar en iframe o nueva pestaña
|
| === CASO 4: Actualizar certificado ===
| 1. Frontend envía nuevo archivo a POST /api/files/update
| 2. Incluye old_file_path con ruta del anterior
| 3. Backend elimina archivo anterior y sube el nuevo
| 4. Actualiza BD con nueva ruta
|
| === CASO 5: Eliminar documento ===
| 1. Frontend solicita DELETE /api/files/delete
| 2. Incluye path del archivo y modelo asociado
| 3. Backend elimina archivo y limpia campo en BD (null)
|
|--------------------------------------------------------------------------
| RESPUESTAS ESTÁNDAR
|--------------------------------------------------------------------------
|
| ÉXITO (upload, update, delete):
| {
|     "success": true,
|     "message": "Archivo [operación] exitosamente",
|     "data": {
|         "path": "students/documents/uuid_archivo.pdf",
|         "file_name": "uuid_archivo.pdf",
|         "original_name": "documento-original.pdf",
|         "extension": "pdf",
|         "size": 245678,
|         "mime_type": "application/pdf",
|         "disk": "local",
|         "updated_record": {  // Solo si se incluyó modelo
|             "model": "Student",
|             "model_id": 123,
|             "field": "document_file"
|         }
|     }
| }
|
| ERROR:
| {
|     "success": false,
|     "message": "Descripción del error",
|     "data": null
| }
|
| VERIFICACIÓN (check):
| {
|     "success": true,
|     "message": "Archivo encontrado" | "Archivo no encontrado",
|     "data": {
|         "exists": true | false,
|         "path": "students/documents/uuid_archivo.pdf",
|         "disk": "local"
|     }
| }
|
|--------------------------------------------------------------------------
| DISCOS DISPONIBLES
|--------------------------------------------------------------------------
|
| 'public':
| - Ubicación física: storage/app/public/
| - Acceso: URL directa /storage/{path}
| - Uso: Fotos de perfil, logos, imágenes públicas
| - Ejemplo: /storage/students/photos/uuid_foto.jpg
|
| 'local':
| - Ubicación física: storage/app/private/
| - Acceso: Solo vía endpoints display/download
| - Uso: Documentos personales, certificados, expedientes
| - Ejemplo: /api/files/display?path=students/documents/uuid_dni.pdf&disk=local
|
|--------------------------------------------------------------------------
| INTEGRACIÓN EN CONTROLLERS ESPECÍFICOS
|--------------------------------------------------------------------------
|
| Puedes copiar estas funciones a tus controllers específicos:
|
| // En StudentController.php
| private FileStorageService $fileService;
|
| public function __construct(FileStorageService $fileService)
| {
|     $this->fileService = $fileService;
| }
|
| public function uploadDocument(Request $request, int $studentId)
| {
|     $student = Student::findOrFail($studentId);
|     
|     // Validar permisos: solo el estudiante puede subir su documento
|     if (auth()->id() !== $student->id) {
|         abort(403, 'No autorizado');
|     }
|     
|     $request->validate(['document' => 'required|file|mimes:pdf|max:5120']);
|     
|     $result = $this->fileService->upload(
|         file: $request->file('document'),
|         disk: 'local',
|         path: 'students/documents',
|         allowedTypes: ['pdf'],
|         maxSize: 5120
|     );
|     
|     if ($result['success']) {
|         $student->update(['document_file' => $result['data']['path']]);
|     }
|     
|     return response()->json($result);
| }
|
| public function viewDocument(int $studentId)
| {
|     $student = Student::findOrFail($studentId);
|     
|     // Validar permisos
|     if (auth()->id() !== $student->id) {
|         abort(403, 'No autorizado');
|     }
|     
|     return $this->fileService->display($student->document_file, 'local');
| }
|
|--------------------------------------------------------------------------
| USO EN REACT/FRONTEND
|--------------------------------------------------------------------------
|
| // 1. Subir archivo con modelo
| const handleUpload = async (file) => {
|     const formData = new FormData();
|     formData.append('file', file);
|     formData.append('disk', 'local');
|     formData.append('path', 'students/documents');
|     formData.append('model', 'Student');
|     formData.append('model_id', studentId);
|     formData.append('field', 'document_file');
|     
|     const response = await fetch('/api/files/upload', {
|         method: 'POST',
|         headers: { 'Authorization': `Bearer ${token}` },
|         body: formData,
|     });
|     
|     const result = await response.json();
|     if (result.success) {
|         console.log('Archivo subido:', result.data.path);
|     }
| };
|
| // 2. Ver archivo privado en iframe
| <iframe 
|     src={`/api/files/display?path=${student.document_file}&disk=local`}
|     width="100%" 
|     height="600px"
| />
|
| // 3. Descargar archivo
| <a 
|     href={`/api/files/download?path=${student.document_file}&disk=local`}
|     download
| >
|     Descargar Documento
| </a>
|
| // 4. Mostrar imagen pública
| <img 
|     src={`${API_URL}/storage/${student.photo_student}`} 
|     alt="Foto"
| />
|
| // 5. Actualizar archivo
| const handleUpdate = async (newFile, oldPath) => {
|     const formData = new FormData();
|     formData.append('file', newFile);
|     formData.append('disk', 'local');
|     formData.append('path', 'students/documents');
|     formData.append('old_file_path', oldPath);
|     formData.append('model', 'Student');
|     formData.append('model_id', studentId);
|     formData.append('field', 'document_file');
|     
|     const response = await fetch('/api/files/update', {
|         method: 'POST',
|         headers: { 'Authorization': `Bearer ${token}` },
|         body: formData,
|     });
| };
|
| // 6. Eliminar archivo
| const handleDelete = async (filePath) => {
|     const response = await fetch('/api/files/delete', {
|         method: 'DELETE',
|         headers: {
|             'Authorization': `Bearer ${token}`,
|             'Content-Type': 'application/json',
|         },
|         body: JSON.stringify({
|             path: filePath,
|             disk: 'local',
|             model: 'Student',
|             model_id: studentId,
|             field: 'document_file',
|         }),
|     });
| };
|
|--------------------------------------------------------------------------
*/