<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\IndexQueryRequest;
use App\Http\Requests\RoleRequest;
use App\Http\Resources\RoleResource;
use App\Models\Permission;
use App\Models\Role;
use Illuminate\Database\QueryException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RoleController extends Controller
{

    private array $messages = [
        'index'          => 'Lista de Roles obtenidos exitosamente',
        'store'          => 'Rol creado exitosamente',
        'show'           => 'Rol obtenido exitosamente',
        'update'         => 'Rol actualizado exitosamente',
        'destroy'        => 'Rol eliminado exitosamente',
        'not_found'      => 'Rol no encontrado',
        'cannot_destroy' => 'No se puede eliminar el Rol porque esta en uso',
        'assign_permission' => 'Permisos asignados exitosamente',
        'remove_permission' => 'Permisos eliminados exitosamente',
        'sync_permission' => 'Permisos sincronizados exitosamente',
    ];

    public function index(IndexQueryRequest $request): JsonResponse
    {
        $items = Role::query()
            ->included()
            ->search()
            ->filter()
            ->sort()
            ->paginated();

        return RoleResource::collection($items)
            ->additional([
                'success' => true,
                'message' => $this->messages['index']
            ])
            ->response();
    }

    public function store(RoleRequest $request): JsonResponse
    {
        $item = Role::create($request->validated());

        return RoleResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['store']
            ])
            ->response()
            ->setStatusCode(201);
    }

    public function show(int $id): JsonResponse
    {
        $item = Role::findOrAbort($id, $this->messages['not_found']);

        return RoleResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['show']
            ])
            ->response();
    }

    public function update(RoleRequest $request, int $id): JsonResponse
    {
        $item = Role::findOrAbort($id, $this->messages['not_found']);
        $item->update($request->validated());

        return RoleResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['update']
            ])
            ->response();
    }

    public function destroy(int $id): JsonResponse
    {
        try {
            $item = Role::findOrAbort($id, $this->messages['not_found']);
            $item->delete();

            return RoleResource::make($item)
                ->additional([
                    'success' => true,
                    'message' => $this->messages['destroy']
                ])
                ->response();
        } catch (QueryException $e) {
            // Foreign key constraint violation
            if ($e->getCode() === '23000') {
                return response()->json([
                    'success' => false,
                    'message' => $this->messages['cannot_destroy']
                ], 409);
            }

            // Re-lanzar otros errores para que el handler global los maneje
            throw $e;
        }
    }

    // Activar o desactivar un rol
    public function toggleStatus(int $id): RoleResource
    {
        $item = Role::findOrAbort($id, $this->messages['not_found']);
        $item->update(['active' => !$item->active]);

        return RoleResource::make($item)->additional([
            'success' => true,
            'message' => $this->messages['update']
        ]);
    }

    // Asignar Permisos a un rol
    public function assignPermissions(Request $request, int $id): JsonResponse | RoleResource
    {
        try {
            $validated = $request->validate([
                'permissions' => 'required|array|min:1',
                'permissions.*' => 'required|exists:permissions,slug',
            ], [], [
                'permissions' => 'Permisos',
            ]);

            $item = Role::findOrAbort($id, $this->messages['not_found']);

            $permissionSlugs = $validated['permissions'];
            // Obtener los IDs de los permisos
            $permissionIds = Permission::whereIn('slug', $permissionSlugs)
                ->pluck('id')
                ->toArray();

            DB::beginTransaction();
            $item->permissions()->syncWithoutDetaching($permissionIds);
            DB::commit();

            return RoleResource::make($item->load('permissions'))->additional([
                'success' => true,
                'message' => $this->messages['assign_permission']
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al asignar permisos: ' . $e->getMessage(),
            ], 500);
        }
    }

    // Eliminar Permisos de un rol
    public function removePermissions(Request $request, int $id): JsonResponse | RoleResource
    {
        try {
            $validated = $request->validate([
                'permissions' => 'required|array|min:1',
                'permissions.*' => 'required|exists:permissions,slug',
            ], [], [
                'permissions' => 'Permisos',
            ]);

            $item = Role::findOrAbort($id, $this->messages['not_found']);

            $permissionSlugs = $validated['permissions'];
            // Obtener los IDs de los permisos
            $permissionIds = Permission::whereIn('slug', $permissionSlugs)
                ->pluck('id')
                ->toArray();

            DB::beginTransaction();
            $item->permissions()->detach($permissionIds);
            DB::commit();

            return RoleResource::make($item->load('permissions'))->additional([
                'success' => true,
                'message' => $this->messages['remove_permission'],
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar permisos: ' . $e->getMessage(),
            ], 500);
        }
    }


    // Sincronizar Permisos de un Rol (puede enviar un array vacio para remover todos)
    public function syncPermissions(Request $request, int $id): JsonResponse | RoleResource
    {
        try {
            $validated = $request->validate([
                'permissions' => 'array',
                'permissions.*' => 'exists:permissions,slug',
            ], [], [
                'permissions' => 'Permisos',
            ]);

            $item = Role::findOrAbort($id, $this->messages['not_found']);

            $permissionSlugs = $validated['permissions'];
            // Obtener los IDs de los permisos
            $permissionIds = Permission::whereIn('slug', $permissionSlugs)
                ->pluck('id')
                ->toArray();

            DB::beginTransaction();
            $item->permissions()->sync($permissionIds ?? []);
            DB::commit();

            return RoleResource::make($item->load('permissions'))->additional([
                'success' => true,
                'message' => $this->messages['sync_permission'],
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al sincronizar permisos: ' . $e->getMessage(),
            ], 500);
        }
    }
}
