<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\IndexQueryRequest;
use App\Http\Requests\ProvinceRequest;
use App\Http\Resources\ProvinceResource;
use App\Models\Province;
use Illuminate\Database\QueryException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ProvinceController extends Controller
{

    private array $messages = [
        'index'          => 'Lista de Provincias obtenidos exitosamente',
        'store'          => 'Provincia creado exitosamente',
        'show'           => 'Provincia obtenido exitosamente',
        'update'         => 'Provincia actualizado exitosamente',
        'destroy'        => 'Provincia eliminado exitosamente',
        'not_found'      => 'Provincia no encontrado',
        'cannot_destroy' => 'No se puede eliminar el Provincia porque tiene distritos asociados',
    ];

    public function index(IndexQueryRequest $request): JsonResponse
    {
        $items = Province::query()
            ->included()
            ->search()
            ->filter()
            ->sort()
            ->paginated();

        return ProvinceResource::collection($items)
            ->additional([
                'success' => true,
                'message' => $this->messages['index']
            ])
            ->response();
    }

    public function store(ProvinceRequest $request): JsonResponse
    {
        $item = Province::create($request->validated());

        return ProvinceResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['store']
            ])
            ->response()
            ->setStatusCode(201);
    }

    public function show(int $id): JsonResponse
    {
        $item = Province::findOrAbort($id, $this->messages['not_found']);

        return ProvinceResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['show']
            ])
            ->response();
    }

    public function update(ProvinceRequest $request, int $id): JsonResponse
    {
        $item = Province::findOrAbort($id, $this->messages['not_found']);
        $item->update($request->validated());

        return ProvinceResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['update']
            ])
            ->response();
    }

    public function destroy(int $id): JsonResponse
    {
        try {
            $item = Province::findOrAbort($id, $this->messages['not_found']);
            $item->delete();

            return ProvinceResource::make($item)
                ->additional([
                    'success' => true,
                    'message' => $this->messages['destroy']
                ])
                ->response();
        } catch (QueryException $e) {
            // Foreign key constraint violation
            if ($e->getCode() === '23000') {
                return response()->json([
                    'success' => false,
                    'message' => $this->messages['cannot_destroy']
                ], 409);
            }

            // Re-lanzar otros errores para que el handler global los maneje
            throw $e;
        }
    }
}
