<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\DiscountTypeRequest;
use App\Http\Requests\IndexQueryRequest;
use App\Http\Resources\DiscountTypeResource;
use App\Models\DiscountType;
use Illuminate\Database\QueryException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class DiscountTypeController extends Controller
{

    private array $messages = [
        'index'          => 'Lista de Tipo de Descuentos obtenidos exitosamente',
        'store'          => 'Tipo de Descuento creado exitosamente',
        'show'           => 'Tipo de Descuento obtenido exitosamente',
        'update'         => 'Tipo de Descuento actualizado exitosamente',
        'destroy'        => 'Tipo de Descuento eliminado exitosamente',
        'not_found'      => 'Tipo de Descuento no encontrado',
        'cannot_destroy' => 'No se puede eliminar el Tipo de Descuento porque esta en uso',
    ];

    public function index(IndexQueryRequest $request): JsonResponse
    {
        $items = DiscountType::query()
            ->included()
            ->search()
            ->filter()
            ->sort()
            ->paginated();

        return DiscountTypeResource::collection($items)
            ->additional([
                'success' => true,
                'message' => $this->messages['index']
            ])
            ->response();
    }

    public function store(DiscountTypeRequest $request): JsonResponse
    {
        $item = DiscountType::create($request->validated());

        return DiscountTypeResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['store']
            ])
            ->response()
            ->setStatusCode(201);
    }

    public function show(int $id): JsonResponse
    {
        $item = DiscountType::findOrAbort($id, $this->messages['not_found']);

        return DiscountTypeResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['show']
            ])
            ->response();
    }

    public function update(DiscountTypeRequest $request, int $id): JsonResponse
    {
        $item = DiscountType::findOrAbort($id, $this->messages['not_found']);
        $item->update($request->validated());

        return DiscountTypeResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['update']
            ])
            ->response();
    }

    public function destroy(int $id): JsonResponse
    {
        try {
            $item = DiscountType::findOrAbort($id, $this->messages['not_found']);
            $item->delete();

            return DiscountTypeResource::make($item)
                ->additional([
                    'success' => true,
                    'message' => $this->messages['destroy']
                ])
                ->response();
        } catch (QueryException $e) {
            // Foreign key constraint violation
            if ($e->getCode() === '23000') {
                return response()->json([
                    'success' => false,
                    'message' => $this->messages['cannot_destroy']
                ], 409);
            }

            // Re-lanzar otros errores para que el handler global los maneje
            throw $e;
        }
    }
}
