<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\CountryRequest;
use App\Http\Requests\IndexQueryRequest;
use App\Http\Resources\CountryResource;
use App\Models\Country;
use Illuminate\Database\QueryException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class CountryController extends Controller
{
    private array $messages = [
        'index'          => 'Lista de Países obtenidos exitosamente',
        'store'          => 'País creado exitosamente',
        'show'           => 'País obtenido exitosamente',
        'update'         => 'País actualizado exitosamente',
        'destroy'        => 'País eliminado exitosamente',
        'not_found'      => 'País no encontrado',
        'cannot_destroy' => 'No se puede eliminar el país porque tiene departamentos asignados',
    ];

    public function index(IndexQueryRequest $request): JsonResponse
    {
        $items = Country::query()
            ->included()
            ->search()
            ->filter()
            ->sort()
            ->paginated();

        return CountryResource::collection($items)
            ->additional([
                'success' => true,
                'message' => $this->messages['index']
            ])
            ->response();
    }

    public function store(CountryRequest $request): JsonResponse
    {
        $item = Country::create($request->validated());

        return CountryResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['store']
            ])
            ->response()
            ->setStatusCode(201);
    }

    public function show(int $id): JsonResponse
    {
        $item = Country::findOrAbort($id, $this->messages['not_found']);

        return CountryResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['show']
            ])
            ->response();
    }

    public function update(CountryRequest $request, int $id): JsonResponse
    {
        $item = Country::findOrAbort($id, $this->messages['not_found']);
        $item->update($request->validated());

        return CountryResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['update']
            ])
            ->response();
    }

    public function destroy(int $id): JsonResponse
    {
        try {
            $item = Country::findOrAbort($id, $this->messages['not_found']);
            $item->delete();

            return CountryResource::make($item)
                ->additional([
                    'success' => true,
                    'message' => $this->messages['destroy']
                ])
                ->response();
        } catch (QueryException $e) {
            // Foreign key constraint violation
            if ($e->getCode() === '23000') {
                return response()->json([
                    'success' => false,
                    'message' => $this->messages['cannot_destroy']
                ], 409);
            }

            // Re-lanzar otros errores para que el handler global los maneje
            throw $e;
        }
    }
}
