<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\CareerRequest;
use App\Http\Requests\IndexQueryRequest;
use App\Http\Resources\CareerResource;
use App\Models\Career;
use App\Traits\ApiQueryTrait;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Database\QueryException;

class CareerController extends Controller
{
    private array $messages = [
        'index'          => 'Lista de Carreras Profesionales obtenida exitosamente',
        'store'          => 'Carrera Profesional creada exitosamente',
        'show'           => 'Carrera Profesional obtenida exitosamente',
        'update'         => 'Carrera Profesional actualizada exitosamente',
        'destroy'        => 'Carrera Profesional eliminada exitosamente',
        'not_found'      => 'Carrera Profesional no encontrada',
        'cannot_destroy' => 'No se puede eliminar la Carrera porque está siendo utilizada',
    ];

    public function index(IndexQueryRequest $request): JsonResponse
    {
        $items = Career::query()
            ->included()
            ->search()
            ->filter()
            ->sort()
            ->paginated();

        return CareerResource::collection($items)
            ->additional([
                'success' => true,
                'message' => $this->messages['index']
            ])
            ->response();
    }

    public function store(CareerRequest $request): JsonResponse
    {
        $item = Career::create($request->validated());

        return CareerResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['store']
            ])
            ->response()
            ->setStatusCode(201);
    }

    public function show(int $id): JsonResponse
    {
        $item = Career::findOrAbort($id, $this->messages['not_found']);

        return CareerResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['show']
            ])
            ->response();
    }

    public function update(CareerRequest $request, int $id): JsonResponse
    {
        $item = Career::findOrAbort($id, $this->messages['not_found']);
        $item->update($request->validated());

        return CareerResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['update']
            ])
            ->response();
    }

    public function destroy(int $id): JsonResponse
    {
        try {
            $item = Career::findOrAbort($id, $this->messages['not_found']);
            $item->delete();

            return CareerResource::make($item)
                ->additional([
                    'success' => true,
                    'message' => $this->messages['destroy']
                ])
                ->response();
        } catch (QueryException $e) {
            // Foreign key constraint violation
            if ($e->getCode() === '23000') {
                return response()->json([
                    'success' => false,
                    'message' => $this->messages['cannot_destroy']
                ], 409);
            }

            // Re-lanzar otros errores para que el handler global los maneje
            throw $e;
        }
    }
}
