<?php

use Illuminate\Http\JsonResponse;

if (!function_exists('auth_response_success')) {
    /**
     * Genera una respuesta exitosa de autenticación
     * 
     * @param array $user Datos del usuario (id, type, email, full_name, etc.)
     * @param string $token Token de autenticación generado
     * @param string $tokenExpires Fecha de expiración del token (ISO 8601)
     * @param array $meta Información adicional (roles, permisos, configuraciones, etc.)
     * @param string $message Mensaje de respuesta personalizado
     * @param int $status Código HTTP de respuesta
     * @return JsonResponse
     */
    function auth_response_success(
        array $user,
        string $token,
        string $tokenExpires,
        array $meta = [],
        string $message = 'Autenticación exitosa',
        int $status = 200
    ): JsonResponse {
        // Convertir a ISO 8601 si viene en otro formato
        $expiresAtFormatted = \Carbon\Carbon::parse($tokenExpires)->toIso8601String();

        return response()->json([
            'success' => true,
            'message' => $message,
            'data' => [
                'user' => $user,
                'access_token' => $token,
                'expires_at' => $expiresAtFormatted,
                'meta' => $meta
            ]
        ], $status);
    }
}

if (!function_exists('auth_response_error')) {
    /**
     * Genera una respuesta de error de autenticación
     * 
     * @param string $message Mensaje de error descriptivo
     * @param int $status Código HTTP de error (401, 403, 422, etc.)
     * @return JsonResponse
     */
    function auth_response_error(
        string $message = 'Error de autenticación',
        int $status = 401
    ): JsonResponse {
        return response()->json([
            'success' => false,
            'message' => $message
        ], $status);
    }
}
