<?php

namespace App\Traits;

use Illuminate\Database\Eloquent\Builder;

trait QueryScopeTrait
{
    /**
     * ========================================
     * INCLUIR RELACIONES
     * ========================================
     * Uso: ?include=user,career,enrollments
     */
    public function scopeIncluded(Builder $query)
    {
        if (empty($this->allowedIncludes) || !request()->filled('include')) {
            return $query;
        }

        $requestedIncludes = array_map('trim', explode(',', request('include')));
        $validIncludes = array_intersect($requestedIncludes, $this->allowedIncludes);

        return !empty($validIncludes) ? $query->with($validIncludes) : $query;
    }

    /**
     * ========================================
     * BÚSQUEDA DE TEXTO
     * ========================================
     * Uso: ?search=Juan Pérez
     */
    public function scopeSearch(Builder $query)
    {
        if (empty($this->searchable) || !request()->filled('search')) {
            return $query;
        }

        $search = trim(request('search'));

        if (empty($search)) {
            return $query;
        }

        return $query->where(function ($q) use ($search) {
            foreach ($this->searchable as $column) {
                $q->orWhere($column, 'LIKE', '%' . $search . '%');
            }
        });
    }

    /**
     * ========================================
     * BÚSQUEDA EN RELACIONES
     * ========================================
     * Uso: ?search=Juan (busca en relaciones definidas en $searchableRelations)
     * 
     * Configuración en modelo:
     * public array $searchableRelations = [
     *     'student' => ['first_name', 'last_name', 'document_number'],
     *     'career' => ['name', 'code'],
     * ];
     */
    public function scopeSearchRelations(Builder $query)
    {
        // Sin búsqueda o sin configuración de relaciones
        if (!request()->filled('search') || empty($this->searchableRelations)) {
            return $query;
        }

        $search = trim(request('search'));

        if (empty($search)) {
            return $query;
        }

        return $query->where(function ($q) use ($search) {
            foreach ($this->searchableRelations as $relation => $columns) {
                $q->orWhereHas($relation, function ($subQuery) use ($columns, $search) {
                    $subQuery->where(function ($sq) use ($columns, $search) {
                        foreach ($columns as $column) {
                            $sq->orWhere($column, 'LIKE', "%{$search}%");
                        }
                    });
                });
            }
        });
    }

    /**
     * ========================================
     * FILTROS ESPECÍFICOS
     * ========================================
     * Uso: ?filter[active]=1&filter[career_id]=5
     */
    public function scopeFilter(Builder $query)
    {
        if (empty($this->filterable) || !request()->filled('filter')) {
            return $query;
        }

        $filters = request('filter');

        if (!is_array($filters)) {
            return $query;
        }

        return $query->where(function ($q) use ($filters) {
            foreach ($filters as $field => $value) {
                if (in_array($field, $this->filterable) && $value !== '') {
                    $q->where($field, $value);
                }
            }
        });
    }

    /**
     * ========================================
     * ORDENAMIENTO
     * ========================================
     * Soporta dos formatos:
     * 
     * FORMATO 1 (moderno): ?sort=-id,name
     * - Prefijo '-' = descendente
     * - Sin prefijo = ascendente
     * - Múltiples columnas separadas por coma
     * 
     * FORMATO 2 (legacy): ?sort_by=name&order=desc
     * - sort_by: columna a ordenar
     * - order: asc o desc
     */
    public function scopeSort(Builder $query)
    {
        // Si no hay columnas ordenables, orden por defecto
        if (empty($this->sortable)) {
            return $query->orderBy('id', 'desc');
        }

        // FORMATO 1: ?sort=-id,name (moderno)
        if (request()->filled('sort')) {
            $sortFields = array_map('trim', explode(',', request('sort')));

            foreach ($sortFields as $sortField) {
                $direction = 'asc';

                // Detectar prefijo '-' para descendente
                if (str_starts_with($sortField, '-')) {
                    $direction = 'desc';
                    $sortField = substr($sortField, 1);
                }

                // Solo ordenar por columnas permitidas
                if (in_array($sortField, $this->sortable)) {
                    $query->orderBy($sortField, $direction);
                }
            }

            return $query;
        }

        // FORMATO 2: ?sort_by=name&order=desc (legacy/compatibilidad)
        if (request()->filled('sort_by')) {
            $sortBy = request('sort_by');
            $order = request('order', 'asc');

            // Validar orden
            if (!in_array($order, ['asc', 'desc'])) {
                $order = 'asc';
            }

            // Solo ordenar por columnas permitidas
            if (in_array($sortBy, $this->sortable)) {
                return $query->orderBy($sortBy, $order);
            }
        }

        // Sin parámetros: orden por defecto
        return $query->orderBy($this->sortable[0] ?? 'id', 'desc');
    }

    /**
     * ========================================
     * PAGINACIÓN
     * ========================================
     * 
     * SIN ?page → Retorna todos (máx 1000)
     * CON ?page → Retorna paginado
     * 
     * Uso: ?page=1&per_page=20
     */
    public function scopePaginated(Builder $query, int $defaultPerPage = 15)
    {
        // Sin paginación: retornar todos (con límite de seguridad)
        if (!request()->has('page')) {
            return $query->limit(1000)->get();
        }

        // Con paginación
        $perPage = (int) request('per_page', $defaultPerPage);
        $perPage = min(max($perPage, 1), 100); // Entre 1 y 100

        return $query->paginate($perPage);
    }

    /**
     * ========================================
     * BUSCAR POR ID O ABORTAR
     * ========================================
     * Uso: Student::findOrAbort($id, 'Estudiante no encontrado')
     */
    public function scopeFindOrAbort(Builder $query, $id, string $message = 'Recurso no encontrado')
    {
        $resource = $query->find($id);

        if (!$resource) {
            throw new \Illuminate\Http\Exceptions\HttpResponseException(
                response()->json([
                    'success' => false,
                    'message' => $message,
                ], 404)
            );
        }

        return $resource;
    }

    /**
     * ========================================
     * BUSCAR POR CAMPO O ABORTAR
     * ========================================
     * Uso: Student::findByOrAbort('email', $email, 'Email no encontrado')
     */
    public function scopeFindByOrAbort(Builder $query, string $field, $value, string $message = 'Recurso no encontrado')
    {
        $resource = $query->where($field, $value)->first();

        if (!$resource) {
            throw new \Illuminate\Http\Exceptions\HttpResponseException(
                response()->json([
                    'success' => false,
                    'message' => $message,
                ], 404)
            );
        }

        return $resource;
    }
}

/**
 * ============================================================
 * QueryScopeTrait - Scopes Reutilizables para Queries Dinámicas
 * ============================================================
 * 
 * CONFIGURACIÓN EN EL MODELO:
 * 
 * public array $searchable = ['name', 'email'];           // Búsqueda de texto
 * public array $sortable = ['id', 'name', 'created_at'];  // Ordenamiento
 * public array $allowedIncludes = ['roles', 'permissions']; // Relaciones
 * public array $filterable = ['active', 'status'];        // Filtros específicos
 * 
 * ============================================================
 * USO EN CONTROLADOR:
 * ============================================================
 * 
 * // LISTAR con todas las funcionalidades:
 * $items = Student::query()
 *     ->included()   // (?include=user,career)
 *     ->search()     // (?search=Juan)
 *     ->filter()     // (?filter[active]=1)
 *     ->sort()       // (?sort=-id) o (?sort_by=name&order=desc)
 *     ->paginated(); // (?page=1&per_page=20)
 * 
 * // BUSCAR por ID:
 * $item = Student::findOrAbort($id, 'Estudiante no encontrado');
 * 
 * // BUSCAR por campo:
 * $item = Student::findByOrAbort('email', $email, 'Email no encontrado');
 * 
 * ============================================================
 * PARÁMETROS DE URL SOPORTADOS:
 * ============================================================
 * 
 * ?search=Juan                    → Busca en columnas searchable
 * ?sort=-id                       → Ordena descendente por id
 * ?sort=name,-created_at          → Ordena múltiples columnas
 * ?sort_by=name&order=desc        → Ordenamiento alternativo (compatibilidad)
 * ?filter[active]=1               → Filtra por campo específico
 * ?filter[career_id]=5            → Múltiples filtros
 * ?include=user,career            → Carga relaciones
 * ?page=1&per_page=20             → Paginación
 * 
 * SIN ?page → Retorna todos (máx 1000)
 * CON ?page → Retorna paginado (máx 100 por página)
 */
