<?php

namespace App\Traits;

use App\Models\Role;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use Illuminate\Support\Collection;

trait HasRoles
{
    /**
     * Relación polimórfica con roles
     * 
     * Permite que User, Teacher y Student tengan roles de forma flexible
     * 
     * @return MorphToMany
     */
    public function roles(): MorphToMany
    {
        return $this->morphToMany(Role::class, 'model', 'model_roles')
            ->withTimestamps();
    }

    /**
     * Asignar uno o más roles al usuario
     * 
     * Los roles se identifican por su slug (identificador único)
     * Si el usuario ya tiene el rol, no lo duplica (syncWithoutDetaching)
     * 
     * @param string|array $roles Slug(s) del/los rol(es) a asignar
     * @throws \Exception Si algún rol no existe
     * @return void
     * 
     * Ejemplo:
     * $user->assignRole('admin');
     * $user->assignRole(['admin', 'coordinator']);
     */
    public function assignRole(string|array $roles): void
    {
        $roles = is_array($roles) ? $roles : [$roles];
        $roleIds = Role::whereIn('slug', $roles)->pluck('id')->toArray();

        if (count($roleIds) !== count($roles)) {
            throw new \Exception('Uno o más roles no existen');
        }

        $this->roles()->syncWithoutDetaching($roleIds);
    }

    /**
     * Remover uno o más roles del usuario
     * 
     * Si el rol no existe o el usuario no lo tiene, no hace nada (no lanza error)
     * 
     * @param string|array $roles Slug(s) del/los rol(es) a remover
     * @return void
     * 
     * Ejemplo:
     * $user->removeRole('admin');
     * $user->removeRole(['admin', 'coordinator']);
     */
    public function removeRole(string|array $roles): void
    {
        $roles = is_array($roles) ? $roles : [$roles];
        $roleIds = Role::whereIn('slug', $roles)->pluck('id')->toArray();

        if (!empty($roleIds)) {
            $this->roles()->detach($roleIds);
        }
    }

    /**
     * Verificar si el usuario tiene uno o más roles específicos
     * 
     * Si se pasa un array, verifica si tiene AL MENOS UNO de los roles (OR lógico)
     * 
     * @param string|array $roles Slug(s) del/los rol(es) a verificar
     * @return bool True si tiene el rol (o al menos uno), False si no
     * 
     * Ejemplo:
     * $user->hasRole('admin') → True/False
     * $user->hasRole(['admin', 'coordinator']) → True si tiene alguno
     */
    public function hasRole(string|array $roles): bool
    {
        if (is_array($roles)) {
            return $this->roles()->whereIn('slug', $roles)->exists();
        }

        return $this->roles()->where('slug', $roles)->exists();
    }

    /**
     * Obtener todos los permisos del usuario (consolidados de todos sus roles)
     * 
     * Si el usuario tiene múltiples roles, combina todos los permisos
     * y elimina duplicados automáticamente
     * 
     * @return Collection Colección de objetos Permission únicos
     * 
     * Ejemplo:
     * $user->permissions() → Collection [Permission, Permission, ...]
     * $user->permissions()->pluck('slug') → ["students.view", "payments.approve", ...]
     */
    public function permissions(): Collection
    {
        return $this->roles()
            ->with('permissions')
            ->get()
            ->pluck('permissions')
            ->flatten()
            ->unique('id')
            ->values();
    }

    /**
     * Verificar si el usuario tiene UN permiso específico
     * 
     * Verifica en todos los roles del usuario si existe el permiso
     * 
     * @param string $permission Slug del permiso a verificar
     * @return bool True si tiene el permiso, False si no
     * 
     * Ejemplo:
     * $user->hasPermission('students.view') → True/False
     * $user->hasPermission('payments.delete') → True/False
     * 
     * Uso común:
     * if ($user->hasPermission('students.delete')) {
     *     // Mostrar botón eliminar
     * }
     */
    public function hasPermission(string $permission): bool
    {
        return $this->permissions()->contains('slug', $permission);
    }

    /**
     * Verificar si el usuario tiene AL MENOS UNO de los permisos especificados
     * 
     * Usa lógica OR: retorna True si tiene cualquiera de los permisos
     * 
     * @param array $permissions Array de slugs de permisos a verificar
     * @return bool True si tiene al menos uno, False si no tiene ninguno
     * 
     * Ejemplo:
     * $user->hasAnyPermission(['students.view', 'students.create'])
     * → True si tiene "students.view" O "students.create" (o ambos)
     * 
     * Uso común:
     * // Mostrar módulo si tiene ALGÚN permiso de estudiantes
     * if ($user->hasAnyPermission(['students.view', 'students.edit', 'students.delete'])) {
     *     // Mostrar módulo de estudiantes
     * }
     */
    public function hasAnyPermission(array $permissions): bool
    {
        $userPermissions = $this->permissions()->pluck('slug')->toArray();

        return !empty(array_intersect($permissions, $userPermissions));
    }

    /**
     * Verificar si el usuario tiene TODOS los permisos especificados
     * 
     * Usa lógica AND: retorna True solo si tiene todos los permisos
     * 
     * @param array $permissions Array de slugs de permisos a verificar
     * @return bool True si tiene todos, False si le falta al menos uno
     * 
     * Ejemplo:
     * $user->hasAllPermissions(['students.view', 'students.create'])
     * → True solo si tiene AMBOS permisos
     * 
     * Uso común:
     * // Acción que requiere múltiples permisos obligatorios
     * if ($user->hasAllPermissions(['payments.view', 'payments.approve'])) {
     *     // Puede aprobar pagos (necesita ver Y aprobar)
     * }
     */
    public function hasAllPermissions(array $permissions): bool
    {
        $userPermissions = $this->permissions()->pluck('slug')->toArray();

        return empty(array_diff($permissions, $userPermissions));
    }

    /**
     * Obtener array simple con los slugs de todos los permisos del usuario
     * 
     * Útil para enviar al frontend (localStorage) o para validaciones rápidas
     * Retorna solo los identificadores (slugs), sin objetos completos
     * 
     * @return array Array de strings con slugs de permisos
     * 
     * Ejemplo:
     * $user->getPermissionSlugs()
     * → ["students.view", "students.create", "payments.approve"]
     * 
     * Uso común en API:
     * return response()->json([
     *     'token' => $token,
     *     'permissions' => $user->getPermissionSlugs(), // Para frontend
     * ]);
     */
    public function getPermissionSlugs(): array
    {
        return $this->permissions()->pluck('slug')->toArray();
    }

    /**
     * Verificar si el usuario tiene todos los roles especificados
     * 
     * Similar a hasAllPermissions pero para roles
     * Usa lógica AND: retorna True solo si tiene todos los roles
     * 
     * @param array $roles Array de slugs de roles a verificar
     * @return bool True si tiene todos los roles, False si le falta al menos uno
     * 
     * Ejemplo:
     * $user->hasAllRoles(['admin', 'coordinator'])
     * → True solo si tiene AMBOS roles
     * 
     * Uso común:
     * if ($user->hasAllRoles(['admin', 'super-admin'])) {
     *     // Usuario con máximo privilegio
     * }
     */
    public function hasAllRoles(array $roles): bool
    {
        $userRoles = $this->roles()->pluck('slug')->toArray();

        return empty(array_diff($roles, $userRoles));
    }

    /**
     * Sincronizar roles (reemplazar todos los roles actuales)
     * 
     * Elimina todos los roles actuales y asigna solo los especificados
     * Diferente a assignRole que agrega sin eliminar los existentes
     * 
     * @param array $roles Array de slugs de roles a sincronizar
     * @throws \Exception Si algún rol no existe
     * @return void
     * 
     * Ejemplo:
     * $user->syncRoles(['admin', 'coordinator']);
     * // Antes: ['admin', 'teacher', 'reviewer']
     * // Después: ['admin', 'coordinator']
     */
    public function syncRoles(array $roles): void
    {
        $roleIds = Role::whereIn('slug', $roles)->pluck('id')->toArray();

        if (count($roleIds) !== count($roles)) {
            throw new \Exception('Uno o más roles no existen');
        }

        $this->roles()->sync($roleIds);
    }

    /**
     * Remover todos los roles del usuario
     * 
     * Deja al usuario sin ningún rol asignado
     * 
     * @return void
     * 
     * Ejemplo:
     * $user->removeAllRoles();
     */
    public function removeAllRoles(): void
    {
        $this->roles()->detach();
    }
}
