<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class PeruDevService
{
    private string $apiUrl;
    private string $apiKey;

    public function __construct()
    {
        $this->apiUrl = config('services.perudev.url');
        $this->apiKey = config('services.perudev.key');
    }

    // Obtener persona por DNI
    public function getPersonByDni(string $dni): array
    {
        try {
            $response = Http::timeout(10)->get($this->apiUrl, [
                'document' => $dni,
                'key' => $this->apiKey,
            ]);

            if (!$response->successful()) {
                Log::error('Error en API PeruDev', [
                    'status' => $response->status(),
                    'dni' => $dni,
                ]);

                return [
                    'success' => false,
                    'message' => 'Error al conectar con el servicio de consulta',
                ];
            }

            $data = $response->json();

            if (!isset($data['estado']) || $data['estado'] === false) {
                return [
                    'success' => false,
                    'message' => $data['mensaje'] ?? 'No se encontró el documento',
                ];
            }

            return [
                'success' => true,
                'message' => 'Encontrado',
                'data' => $this->formatToDatabase($data['resultado']),
            ];
        } catch (\Exception $e) {
            Log::error('Excepción en PeruDevService', [
                'dni' => $dni,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => 'Error al conectar con el servicio de consulta',
            ];
        }
    }

    // Formatear datos a formato de base de datos
    private function formatToDatabase(array $reniecData): array
    {
        return [
            'document_number' => $reniecData['id'] ?? '',
            'names' => $reniecData['nombres'] ?? '',
            'paternal_surname' => $reniecData['apellido_paterno'] ?? '',
            'maternal_surname' => $reniecData['apellido_materno'] ?? '',
            'birth_date' => $this->formatBirthDate($reniecData['fecha_nacimiento'] ?? null),
            'gender' => $this->formatGender($reniecData['genero'] ?? null),
        ];
    }

    // Formatear fecha
    private function formatBirthDate(?string $date): ?string
    {
        if (!$date) {
            return null;
        }

        try {
            $dateTime = \DateTime::createFromFormat('d/m/Y', $date);
            return $dateTime ? $dateTime->format('Y-m-d') : '';
        } catch (\Exception $e) {
            Log::warning('Error formateando fecha', ['date' => $date]);
            return '';
        }
    }

    // Formatear género
    private function formatGender(?string $gender): ?string
    {
        if (!$gender) {
            return null;
        }

        return match (strtoupper($gender)) {
            'M' => 'masculino',
            'F' => 'femenino',
            default => '',
        };
    }
}
