<?php

namespace App\Models;

use App\Enums\PaymentStatus;
use App\Traits\QueryScopeTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Payment extends Model
{
    use HasFactory, QueryScopeTrait;

    protected $fillable = [
        'method',
        'concept',
        'enrollment_id',
        'payment_date',
        'amount',
        'receipt_number',
        'document_number',
        'full_name',
        'comment',
        'file',
        'status',
        'approved_by',
    ];

    protected $casts = [
        'status' => PaymentStatus::class,
        'payment_date' => 'date',
        'amount' => 'decimal:2',
    ];

    // Columnas donde se puede buscar (texto general)
    public array $searchable = ['receipt_number', 'document_number', 'full_name'];

    // Columnas por las que se puede ordenar
    public array $sortable = ['id', 'payment_date', 'amount', 'receipt_number', 'document_number', 'full_name', 'comment'];

    // Relaciones que se pueden incluir
    public array $allowedIncludes = ['approvedBy', 'enrollment'];

    // Columnas por las que se puede filtrar específicamente
    public array $filterable = ['receipt_number', 'concept', 'document_number', 'full_name'];

    public function scopeApprovedEnrollment($query)
    {
        return $query->where('concept', 'Enrollment')
            ->where('status', PaymentStatus::Approved->value);
    }

    public function scopePendingEnrollment($query)
    {
        return $query->where('concept', 'Enrollment')
            ->where('status', PaymentStatus::Pending->value);
    }

    public function approvedBy()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function enrollment()
    {
        return $this->belongsTo(Enrollment::class);
    }
}
