<?php

namespace App\Http\Requests;

use App\Enums\Gender;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class UserRequest extends FormRequest
{
    private ?int $user_id = null;

    // Autorizar todas las peticiones (validación de permisos en middleware)
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        $this->user_id = $this->route("id"); 

        switch ($this->route()->getActionMethod()) {
            case 'login':
                return $this->loginUserRules();
            case 'store':
                return $this->storeUserRules();
            case 'update':
                return $this->updateUserRules();
            default:
                return [];
        }
    }

    private function loginUserRules(): array
    {
        return [
            'document_number' => ['required', 'string', 'min:8', 'max:20'],
            'password' => ['required', 'string', 'min:8', 'max:50'],
        ];
    }

    private function storeUserRules(): array
    {
        return [
            'document_number' => ['required', 'string', 'min:8', 'max:20', 'unique:users,document_number'],
            'names' => ['required', 'string', 'min:3', 'max:50'],
            'paternal_surname' => ['required', 'string', 'min:3', 'max:50'],
            'maternal_surname' => ['required', 'string', 'min:3', 'max:50'],
            'gender' => ['required', Rule::enum(Gender::class)],
            'birth_date' => ['required', 'date', 'before:today'],
            'email' => ['required', 'email', 'max:255', 'unique:users,email'],
            'phone_number' => ['required', 'string', 'max:20'],
            'photo_user' => ['nullable', 'string', 'max:255'],
            'active' => ['required', 'boolean'],
        ];
    }

    private function updateUserRules(): array
    {
        return [
            'document_number' => ['required', 'string', 'min:8', 'max:20', Rule::unique('users', 'document_number')->ignore($this->user_id)],
            'names' => ['required', 'string', 'min:3', 'max:50'],
            'paternal_surname' => ['required', 'string', 'min:3', 'max:50'],
            'maternal_surname' => ['required', 'string', 'min:3', 'max:50'],
            'gender' => ['required', Rule::enum(Gender::class)],
            'birth_date' => ['required', 'date', 'before:today'],
            'email' => ['required', 'email', 'max:255', Rule::unique('users', 'email')->ignore($this->user_id)],
            'phone_number' => ['required', 'string', 'max:20'],
            'photo_user' => ['nullable', 'string', 'max:255'],
            'active' => ['required', 'boolean'],
        ];
    }

    // Etiquetas de validación
    public function attributes(): array
    {
        return [
            'document_number' => 'Documento',
            'password' => 'Contraseña',
            'names' => 'Nombres',
            'paternal_surname' => 'Apellido paterno',
            'maternal_surname' => 'Apellido materno',
            'gender' => 'Genero',
            'birth_date' => 'Fecha de nacimiento',
            'email' => 'Correo',
            'phone_number' => 'Teléfono',
            'photo_user' => 'Foto',
        ];
    }
}
