<?php

namespace App\Http\Requests;

use App\Enums\DocumentType;
use App\Enums\Gender;
use App\Enums\MaritalStatus;
use App\Traits\AppHelpers;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use PhpParser\Comment\Doc;

class StudentRequest extends FormRequest
{
    use AppHelpers;

    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        // Detecta qué función del controlador se está ejecutando
        switch ($this->route()->getActionMethod()) {
            // Student Auth Controller (Student)
            case 'checkStudent':
                return $this->checkStudentRules();
            case 'registerInformation':
                return $this->registerInformationRules();
            case 'login':
                return $this->loginInformationRules();
            case 'changePassword':
                return $this->changePasswordRules();
                // Student Controller (Admin)
            case 'store':
                return $this->storeInformationRules();
            case 'update':
                return $this->updateInformationRules();
            default:
                return [];
        }
    }

    //  ============================================================
    //  Student Auth Controller (Para estudiantes)
    //  ============================================================

    private function checkStudentRules(): array
    {
        return [
            'document_type' => ['required', 'in:dni,passport,foreigner_card'],
            'document_number' => ['required', 'string', 'min:8', 'max:20'],
        ];
    }

    private function registerInformationRules(): array
    {
        return [
            'document_type' => ['required', 'in:dni,passport,foreigner_card'],
            'document_number' => ['required', 'string', 'min:8', 'max:20'],
            'names' => ['required', 'string', 'min:2', 'max:255'],
            'paternal_surname' => ['required', 'string', 'min:2', 'max:255'],
            'maternal_surname' => ['required', 'string', 'min:2', 'max:255'],
            'birth_date' => ['required', 'date', 'before:today'],
            'gender' => ['required', 'in:male,female'],
            'phone_number' => ['required', 'string', 'size:9'],
            'whatsapp_number' => ['required', 'string', 'size:9'],
            'email' => ['required', 'email', 'max:255'],
            'password' => ['required', 'string', 'min:8'],
            'cycle' => ['sometimes', 'array'],
        ];
    }

    private function loginInformationRules(): array
    {
        return [
            'document_number' => ['required', 'string', 'min:8', 'max:20'],
            'password' => ['required', 'string', 'min:8', 'max:50'],
        ];
    }

    private function changePasswordRules(): array
    {
        return [
            'password' => ['required', 'string', 'min:8', 'max:50'],
        ];
    }

    //  ============================================================
    //  StudentController (Para Administradores)
    //  ============================================================

    private function storeInformationRules(): array
    {
        return [
            'document_type' => ['required', Rule::enum(DocumentType::class)],
            'document_number' => ['required', 'string', 'min:8', 'max:20', 'unique:students,document_number'],
            'names' => ['required', 'string', 'min:2', 'max:255'],
            'paternal_surname' => ['required', 'string', 'min:2', 'max:255'],
            'maternal_surname' => ['required', 'string', 'min:2', 'max:255'],
            'birth_date' => ['required', 'date', 'before:today'],
            'birth_place_id' => ['required', 'exists:districts,id'],
            'marital_status' => ['required', Rule::enum(MaritalStatus::class)],
            'gender' => ['required', Rule::enum(Gender::class)],
            'languages' => ['required', 'array'],
            'phone_number' => ['required', 'string', 'size:9'],
            'whatsapp_number' => ['required', 'string', 'size:9'],
            'email' => ['required', 'email', 'max:255', 'unique:students,email'],
            'residence_address' => ['required', 'string', 'max:255'],
            'residence_place_id' => ['required', 'exists:districts,id'],
            'school_graduation_year' => ['nullable', 'year', 'before:today'],
            'school_id' => ['required', 'exists:schools,id'],
            'family_data' => ['nullable', 'array'],
            'active' => ['required', 'boolean'],
            'validated_information' => ['required', 'boolean'],
        ];
    }

    private function updateInformationRules(): array
    {
        $student_id = $this->route('id');
        return [
            'document_type' => ['required', Rule::enum(DocumentType::class)],
            'document_number' => ['required', 'string', 'min:8', 'max:20', Rule::unique('students', 'document_number')->ignore($student_id)],
            'names' => ['required', 'string', 'min:2', 'max:255'],
            'paternal_surname' => ['required', 'string', 'min:2', 'max:255'],
            'maternal_surname' => ['required', 'string', 'min:2', 'max:255'],
            'birth_date' => ['required', 'date', 'before:today'],
            'birth_place_id' => ['required', 'exists:districts,id'],
            'marital_status' => ['required', Rule::enum(MaritalStatus::class)],
            'gender' => ['required', Rule::enum(Gender::class)],
            'languages' => ['required', 'array'],
            'residence_address' => ['required', 'string', 'min:2', 'max:255'],
            'residence_place_id' => ['required', 'exists:districts,id'],
            'phone_number' => ['required', 'string', 'size:9'],
            'whatsapp_number' => ['required', 'string', 'size:9'],
            'email' => ['required', 'email', 'max:255', Rule::unique('students', 'email')->ignore($student_id)],
            'school_graduation_year' => ['nullable', 'numeric', 'before:today'],
            'school_id' => ['nullable', 'exists:schools,id'],
            'family_data' => ['nullable', 'array'],
        ];
    }

    // Preparar la solicitud antes de la validación
    protected function prepareForValidation()
    {
        // Solo aplicar ensureActiveCycle a métodos específicos
        $methodsWithCycle = ['registerInformation'];

        if (in_array($this->route()->getActionMethod(), $methodsWithCycle)) {
            $this->ensureActiveCycle();
        }
    }

    // Definir los nombres de atributos personalizados
    public function attributes(): array
    {
        return [
            'document_type' => 'Tipo de Documento',
            'document_number' => 'Número de Documento',
            'names' => 'Nombres Completos',
            'paternal_surname' => 'Apellido Paterno',
            'maternal_surname' => 'Apellido Materno',
            'birth_date' => 'Fecha de Nacimiento',
            'birth_place_id' => 'Lugar de Nacimiento',
            'marital_status' => 'Estado Civil',
            'gender' => 'Genero',
            'languages' => 'Idiomas',
            'phone_number' => 'Número de Teléfono',
            'whatsapp_number' => 'Número de Whatsapp',
            'email' => 'Correo Electrónico',
            'residence_address' => 'Dirección de Residencia',
            'residence_place_id' => 'Lugar de Residencia',
            'school_graduation_year' => 'Año de Egreso',
            'school_graduation_certificate' => 'Certificado de Egreso',
            'school_id' => 'Colegio',
            'family_data' => 'Datos Familiares',
            'document_file' => 'Archivo de Documento',
            'photo_student' => 'Foto del Estudiante',
            'active' => 'Activo',
            'validated' => 'Validado',
            'password' => 'Contraseña',
            'access_token' => 'Token de Acceso',
            'token_expires_at' => 'Fecha de Expiración del Token',
            'last_login_at' => 'Fecha de Ultimo Ingreso',
            'information_sources' => 'Medios de Información',
            'sworn_declaration' => 'Declaración Jurada',
        ];
    }
}
