<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class CheckPermission
{
    /**
     * Middleware para verificar permisos del usuario autenticado.
     * 
     * Se ejecuta DESPUÉS de los middlewares de autenticación (auth.admin, auth.teacher, auth.student).
     * Verifica si el usuario tiene AL MENOS UNO de los permisos especificados.
     * 
     * Uso en rutas:
     * ->middleware('permission:students.view')
     * ->middleware('permission:students.edit,students.delete')
     * 
     * @param Request $request
     * @param Closure $next
     * @param string ...$permissions Permisos requeridos (uno o varios separados por coma)
     * @return Response
     */
    public function handle(Request $request, Closure $next, string ...$permissions): Response
    {
        // Obtener usuario autenticado (ya inyectado por BaseAuth)
        $user = $request->user();

        // Si no hay usuario autenticado, retornar error 401
        if (!$user) {
            return $this->forbiddenResponse('Usuario no autenticado');
        }

        // Si no se especificaron permisos, permitir acceso (configuración incorrecta)
        if (empty($permissions)) {
            return $next($request);
        }

        // Verificar si el usuario tiene AL MENOS UNO de los permisos requeridos
        if ($user->hasAnyPermission($permissions)) {
            return $next($request);
        }

        // Usuario no tiene ninguno de los permisos requeridos
        return $this->forbiddenResponse(
            'No tienes permisos para realizar esta acción',
            $permissions
        );
    }

    /**
     * Respuesta estandarizada para errores de autorización (403 Forbidden).
     * 
     * @param string $message Mensaje de error
     * @param array $requiredPermissions Permisos que se requerían (opcional, para debug)
     * @return Response
     */
    private function forbiddenResponse(string $message, array $requiredPermissions = []): Response
    {
        $response = [
            'success' => false,
            'message' => $message,
        ];

        // En desarrollo, agregar permisos requeridos para facilitar debug
        if (config('app.debug') && !empty($requiredPermissions)) {
            $response['required_permissions'] = $requiredPermissions;
        }

        return response()->json($response, 403);
    }
}
