<?php

namespace App\Http\Middleware;

use App\Models\Student;
use App\Models\Teacher;
use App\Models\User;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class AuthenticateAny
{
    /**
     * Verifica si el usuario está autenticado con AL MENOS UNO de los guards especificados.
     * 
     * Uso en rutas:
     * ->middleware('auth.any:admin,teacher,student')
     */
    public function handle(Request $request, Closure $next, string ...$guards): Response
    {
        $token = $request->bearerToken();

        if (!$token) {
            return $this->unauthorizedResponse('Token de autenticación requerido');
        }

        // Mapeo de guards a modelos
        $guardModels = [
            'admin' => User::class,
            'teacher' => Teacher::class,
            'student' => Student::class,
        ];

        $hashedToken = hash('sha256', $token);

        // Filtrar solo guards válidos
        $validModels = collect($guards)
            ->filter(fn($guard) => isset($guardModels[$guard]))
            ->map(fn($guard) => $guardModels[$guard])
            ->values()
            ->toArray();

        if (empty($validModels)) {
            return $this->unauthorizedResponse('Configuración de guards inválida');
        }

        // Buscar en todos los modelos en paralelo (una sola iteración optimizada)
        foreach ($validModels as $modelClass) {
            $user = $modelClass::where('access_token', $hashedToken)
                ->where('token_expires_at', '>', now())
                ->first();

            if ($user) {
                // Usuario encontrado, autenticar y continuar
                $user->updateLastLogin();
                $request->setUserResolver(fn() => $user);

                return $next($request);
            }
        }

        // No se pudo autenticar con ningún guard
        return $this->unauthorizedResponse('Token inválido o expirado');
    }

    private function unauthorizedResponse(string $message): Response
    {
        return response()->json([
            'success' => false,
            'message' => $message,
        ], 401);
    }
}
