<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\TeacherRequest;
use App\Http\Resources\TeacherResource;
use App\Models\Teacher;
use App\Traits\ApiQueryTrait;
use Illuminate\Database\QueryException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class TeacherController extends Controller
{
    use ApiQueryTrait;

    private array $messages = [
        'index'          => 'Lista de Docentes obtenidos exitosamente',
        'store'          => 'Docente creado exitosamente',
        'show'           => 'Docente obtenido exitosamente',
        'update'         => 'Docente actualizado exitosamente',
        'destroy'        => 'Docente eliminado exitosamente',
        'not_found'      => 'Docente no encontrado',
        'cannot_destroy' => 'No se puede eliminar el Docente porque esta en uso',
    ];

    public function index(Request $request): JsonResponse
    {
        $items = $this->applyQuery(Teacher::class, $request);

        return TeacherResource::collection($items)
            ->additional([
                'success' => true,
                'message' => $this->messages['index']
            ])
            ->response();
    }

    public function store(TeacherRequest $request): JsonResponse
    {
        $item = Teacher::create($request->validated());

        return TeacherResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['store']
            ])
            ->response()
            ->setStatusCode(201);
    }

    public function show(int $id): JsonResponse
    {
        $item = Teacher::findOrAbort($id, $this->messages['not_found']);

        return TeacherResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['show']
            ])
            ->response();
    }

    public function update(TeacherRequest $request, int $id): JsonResponse
    {
        $item = Teacher::findOrAbort($id, $this->messages['not_found']);
        $item->update($request->validated());

        return TeacherResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['update']
            ])
            ->response();
    }

    public function destroy(int $id): JsonResponse
    {
        try {
            $item = Teacher::findOrAbort($id, $this->messages['not_found']);
            $item->delete();

            return TeacherResource::make($item)
                ->additional([
                    'success' => true,
                    'message' => $this->messages['destroy']
                ])
                ->response();
        } catch (QueryException $e) {
            // Foreign key constraint violation
            if ($e->getCode() === '23000') {
                return response()->json([
                    'success' => false,
                    'message' => $this->messages['cannot_destroy']
                ], 409);
            }

            // Re-lanzar otros errores para que el handler global los maneje
            throw $e;
        }
    }

    // Toggle status
    public function toggleStatus(int $id): TeacherResource
    {
        $item = Teacher::findOrAbort($id, $this->messages['not_found']);
        $item->update(['active' => !$item->active]);

        return TeacherResource::make($item)->additional([
            'success' => true,
            'message' => 'Estado del docente actualizado exitosamente',
        ]);
    }

    // Reset password
    public function resetPassword(int $id): TeacherResource
    {
        $item = Teacher::findOrAbort($id, $this->messages['not_found']);
        $item->update(['password' => $item->document_number]);

        return TeacherResource::make($item)->additional([
            'success' => true,
            'message' => 'Password del docente actualizado exitosamente',
        ]);
    }

    // Cambiar password
    public function changePassword(Request $request, int $id): TeacherResource
    {
        $validated = $request->validate([
            'password' => ['required', 'string', 'min:8'],
        ]);

        $item = Teacher::findOrAbort($id, $this->messages['not_found']);
        $item->update(['password' => $validated['password']]);

        return TeacherResource::make($item)->additional([
            'success' => true,
            'message' => 'Password del docente actualizado exitosamente',
        ]);
    }
}
