<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;

use App\Http\Requests\CycleRequest;
use App\Http\Requests\IndexQueryRequest;
use App\Http\Resources\CycleResource;
use App\Models\Cycle;
use App\Traits\AppHelpers;
use Illuminate\Database\QueryException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class CycleController extends Controller
{
    use AppHelpers;

    private array $messages = [
        'index'          => 'Lista de Ciclos Académicos obtenidos exitosamente',
        'store'          => 'Ciclo Académico creado exitosamente',
        'show'           => 'Ciclo Académico obtenido exitosamente',
        'update'         => 'Ciclo Académico actualizado exitosamente',
        'destroy'        => 'Ciclo Académico eliminado exitosamente',
        'not_found'      => 'Ciclo Académico no encontrado',
        'cannot_destroy' => 'No se puede eliminar el Ciclo Académico porque esta en uso',
        'toggle_enrollment' => 'Inscripciones :state',
        'toggle_active'     => 'Ciclo Académico :state',
    ];

    // Verificar si hay ciclo Activo
    public function checkActiveCycle()
    {
        $cycle =  $this->getActiveCycle();

        return CycleResource::make($cycle)->additional([
            'success' => true,
            'message' => $this->messages['show']
        ]);
    }

    public function index(IndexQueryRequest $request): JsonResponse
    {
        $items = Cycle::query()
            ->included()
            ->search()
            ->filter()
            ->sort()
            ->paginated();

        return CycleResource::collection($items)
            ->additional([
                'success' => true,
                'message' => $this->messages['index']
            ])
            ->response();
    }

    public function store(CycleRequest $request): JsonResponse
    {
        // Verificar si hay ciclo Activo
        $cycle = Cycle::getLastActive();

        if ($cycle) {
            return response()->json([
                'success' => false,
                'message' => 'Ya existe un Ciclo Activo',
            ], 400);
        }

        // Proceder a crear
        $item = Cycle::create($request->validated());

        return CycleResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['store']
            ])
            ->response()
            ->setStatusCode(201);
    }

    public function show(int $id): JsonResponse
    {
        $item = Cycle::findOrAbort($id, $this->messages['not_found']);

        return CycleResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['show']
            ])
            ->response();
    }

    public function update(CycleRequest $request, int $id): JsonResponse
    {
        $item = Cycle::findOrAbort($id, $this->messages['not_found']);
        $item->update($request->validated());

        return CycleResource::make($item)
            ->additional([
                'success' => true,
                'message' => $this->messages['update']
            ])
            ->response();
    }

    public function destroy(int $id): JsonResponse
    {
        try {
            $item = Cycle::findOrAbort($id, $this->messages['not_found']);
            $item->delete();

            return CycleResource::make($item)
                ->additional([
                    'success' => true,
                    'message' => $this->messages['destroy']
                ])
                ->response();
        } catch (QueryException $e) {
            // Foreign key constraint violation
            if ($e->getCode() === '23000') {
                return response()->json([
                    'success' => false,
                    'message' => $this->messages['cannot_destroy']
                ], 409);
            }

            // Re-lanzar otros errores para que el handler global los maneje
            throw $e;
        }
    }

    // Estado Inscripciones
    public function toggleEnrollment(int $id): CycleResource
    {
        $item = Cycle::findOrAbort($id, $this->messages['not_found']);
        $item->update(['enrollment_open' => !$item->enrollment_open]);

        $stateText = $item->enrollment_open ? 'Abiertas' : 'Cerradas';

        $message = str_replace(':state', $stateText, $this->messages['toggle_enrollment']);

        return CycleResource::make($item)->additional([
            'success' => true,
            'message' => $message,
        ]);
    }

    // Estado Cyclos
    public function toggleActive(int $id): CycleResource | JsonResponse
    {
        $item = Cycle::findOrAbort($id, $this->messages['not_found']);

        // Verificar si hay ciclo Activo
        $cycle = Cycle::getLastActive();

        if ($cycle && $item->active === false) {
            return response()->json([
                'success' => false,
                'message' => 'Ya existe un Ciclo Activo',
            ], 400);
        }

        // Proceder a actualizar
        $item->update(['active' => !$item->active]);

        $stateText = $item->active ? 'ACTIVO' : 'CERRADO';

        $message = str_replace(':state', $stateText, $this->messages['toggle_active']);

        return CycleResource::make($item)->additional([
            'success' => true,
            'message' => $message,
        ]);
    }
}
